package ru.yandex.intranet.d.datasource.model;

import java.util.function.BiFunction;
import java.util.function.Function;

import com.yandex.ydb.table.transaction.TransactionMode;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.ResultTx;

/**
 * Chain of Ydb queries.
 * Все запросы выполняются в одной транзакции. Первый запрос стартует транзакцию, последний её коммитит.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 18.09.2020
 */
public interface YdbQueryChain<I, O> {
    static <I, O> YdbQueryChain<I, O> start(Stage<I, O> preamble) {
        return new YdbQueryChainImpl<>(preamble);
    }

    static <I, O> YdbQueryChain<I, O> start(Function<YdbTxSession, Mono<WithTxId<O>>> query) {
        return new YdbQueryChainImpl<>((ts, unused) -> query.apply(ts));
    }

    Mono<O> execute(YdbSession session, TransactionMode txMode, I input);

    Mono<Result<O>> validateAndExecute(YdbSession session, TransactionMode txMode, I input, Validator<I> validator);

    /**
     * Объединяет старые body и trailer в новый body, а добавляемый запрос становится новым trailer.
     * Требуется, чтобы во всех запросах коммит всегда зависел от флага в транзакции.
     */
    <T> YdbQueryChain<I, T> append(Stage<O, T> stage);

    /**
     * Объединяет старые body и trailer, и из добавляемой цепочки preamble и body -- в новый body.
     * A новым trailer становится trailer из добавляемой цепочки.
     * Требуется, чтобы во всех запросах коммит всегда зависел от флага в транзакции.
     */
    <T> YdbQueryChain<I, T> append(YdbQueryChain<O, T> chain);

    <I1, O1, T> YdbQueryChain<I, T> combine(
            YdbQueryChain<I1, O1> chain,
            Function<O, I1> glue,
            BiFunction<O, O1, T> combinator
    );

    interface Stage<I, O> extends BiFunction<YdbTxSession, I, Mono<WithTxId<O>>> {
    }

    interface Validator<I> extends BiFunction<YdbTxSession, I, Mono<ResultTx<I>>> {
    }
}
