package ru.yandex.intranet.d.datasource.model;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;

/**
 * YDB read table settings.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YdbReadTableSettings {

    private final boolean ordered;
    private final TupleValue fromKey;
    private final boolean fromInclusive;
    private final TupleValue toKey;
    private final boolean toInclusive;
    private final int rowLimit;
    private final List<String> columns;
    private final Duration timeout;

    @SuppressWarnings("ParameterNumber")
    private YdbReadTableSettings(boolean ordered, TupleValue fromKey, boolean fromInclusive, TupleValue toKey,
                                boolean toInclusive, int rowLimit, List<String> columns, Duration timeout) {
        this.ordered = ordered;
        this.fromKey = fromKey;
        this.fromInclusive = fromInclusive;
        this.toKey = toKey;
        this.toInclusive = toInclusive;
        this.rowLimit = rowLimit;
        this.columns = columns;
        this.timeout = timeout;
    }

    public static Builder builder() {
        return new Builder();
    }

    public boolean isOrdered() {
        return ordered;
    }

    public Optional<TupleValue> getFromKey() {
        return Optional.ofNullable(fromKey);
    }

    public boolean isFromInclusive() {
        return fromInclusive;
    }

    public Optional<TupleValue> getToKey() {
        return Optional.ofNullable(toKey);
    }

    public boolean isToInclusive() {
        return toInclusive;
    }

    public int getRowLimit() {
        return rowLimit;
    }

    public List<String> getColumns() {
        return columns;
    }

    public Optional<Duration> getTimeout() {
        return Optional.ofNullable(timeout);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YdbReadTableSettings that = (YdbReadTableSettings) o;
        return ordered == that.ordered &&
                fromInclusive == that.fromInclusive &&
                toInclusive == that.toInclusive &&
                rowLimit == that.rowLimit &&
                Objects.equals(fromKey, that.fromKey) &&
                Objects.equals(toKey, that.toKey) &&
                Objects.equals(columns, that.columns) &&
                Objects.equals(timeout, that.timeout);
    }

    @Override
    public int hashCode() {
        return Objects.hash(ordered, fromKey, fromInclusive, toKey, toInclusive, rowLimit, columns, timeout);
    }

    @Override
    public String toString() {
        return "YdbReadTableSettings{" +
                "ordered=" + ordered +
                ", fromKey=" + fromKey +
                ", fromInclusive=" + fromInclusive +
                ", toKey=" + toKey +
                ", toInclusive=" + toInclusive +
                ", rowLimit=" + rowLimit +
                ", columns=" + columns +
                ", timeout=" + timeout +
                '}';
    }

    public static final class Builder {

        private final List<String> columns = new ArrayList<>();

        private boolean ordered;
        private TupleValue fromKey;
        private boolean fromInclusive;
        private TupleValue toKey;
        private boolean toInclusive;
        private int rowLimit;
        private Duration timeout;

        private Builder() {
        }

        public Builder ordered(boolean ordered) {
            this.ordered = ordered;
            return this;
        }

        public Builder fromKey(TupleValue value, boolean inclusive) {
            this.fromKey = value;
            this.fromInclusive = inclusive;
            return this;
        }

        public Builder toKey(TupleValue value, boolean inclusive) {
            this.toKey = value;
            this.toInclusive = inclusive;
            return this;
        }

        public Builder fromKeyInclusive(TupleValue value) {
            return fromKey(value, true);
        }

        public Builder fromKeyExclusive(TupleValue value) {
            return fromKey(value, false);
        }

        public Builder fromKeyInclusive(PrimitiveValue value) {
            return fromKey(TupleValue.of(value.makeOptional()), true);
        }

        public Builder fromKeyExclusive(PrimitiveValue value) {
            return fromKey(TupleValue.of(value.makeOptional()), false);
        }

        public Builder toKeyInclusive(TupleValue value) {
            return toKey(value, true);
        }

        public Builder toKeyExclusive(TupleValue value) {
            return toKey(value, false);
        }

        public Builder toKeyInclusive(PrimitiveValue value) {
            return toKey(TupleValue.of(value.makeOptional()), true);
        }

        public Builder toKeyExclusive(PrimitiveValue value) {
            return toKey(TupleValue.of(value.makeOptional()), false);
        }

        public Builder rowLimit(int rowLimit) {
            this.rowLimit = rowLimit;
            return this;
        }

        public Builder addColumn(String column) {
            this.columns.add(column);
            return this;
        }

        public Builder addColumns(String... columns) {
            this.columns.addAll(List.of(columns));
            return this;
        }

        public Builder addColumns(List<String> columns) {
            this.columns.addAll(columns);
            return this;
        }

        public Builder timeout(long duration, TimeUnit unit) {
            this.timeout = Duration.ofNanos(unit.toNanos(duration));
            return this;
        }

        public Builder timeout(Duration duration) {
            this.timeout = duration;
            return this;
        }

        public YdbReadTableSettings build() {
            return new YdbReadTableSettings(ordered, fromKey, fromInclusive, toKey,
                    toInclusive, rowLimit, columns, timeout);
        }

    }

}
