package ru.yandex.intranet.d.grpc.interceptors;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Preconditions;
import io.grpc.ServerCall;

/**
 * Delaying listener for service calls, queues incoming calls until actual listener is available.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class DelayedServerCallListener<ReqT> extends ServerCall.Listener<ReqT> {

    private volatile boolean passThrough;
    private ServerCall.Listener<ReqT> realListener;
    private List<Runnable> pendingCalls = new ArrayList<>();
    private Runnable cancelCallback;

    @Override
    public void onMessage(ReqT message) {
        if (passThrough) {
            realListener.onMessage(message);
        } else {
            delayOrExecute(() -> realListener.onMessage(message));
        }
    }

    @Override
    public void onHalfClose() {
        if (passThrough) {
            realListener.onHalfClose();
        } else {
            delayOrExecute(() -> realListener.onHalfClose());
        }
    }

    @Override
    public void onCancel() {
        boolean delegateToRealStream = true;
        Runnable cancelCallbackToRun = null;
        synchronized (this) {
            if (realListener == null) {
                setRealListener(new ServerCall.Listener<>() { });
                delegateToRealStream = false;
                cancelCallbackToRun = cancelCallback;
            }
        }
        if (delegateToRealStream) {
            delayOrExecute(() -> realListener.onCancel());
        } else {
            if (cancelCallbackToRun != null) {
                cancelCallbackToRun.run();
            }
            drainPendingCalls();
        }
    }

    @Override
    public void onComplete() {
        if (passThrough) {
            realListener.onComplete();
        } else {
            delayOrExecute(() -> realListener.onComplete());
        }
    }

    @Override
    public void onReady() {
        if (passThrough) {
            realListener.onReady();
        } else {
            delayOrExecute(() -> realListener.onReady());
        }
    }

    public void setCancelCallback(Runnable cancelCallback) {
        Preconditions.checkState(this.cancelCallback == null, "cancel callback already set");
        synchronized (this) {
            this.cancelCallback = Preconditions.checkNotNull(cancelCallback, "cancelCallback");
        }
    }

    public void setListener(ServerCall.Listener<ReqT> listener) {
        synchronized (this) {
            if (realListener != null) {
                return;
            }
            setRealListener(Preconditions.checkNotNull(listener, "listener"));
        }
        drainPendingCalls();
    }

    private void setRealListener(ServerCall.Listener<ReqT> realListener) {
        Preconditions.checkState(this.realListener == null, "realListener already set to %s", this.realListener);
        this.realListener = realListener;
    }

    private void drainPendingCalls() {
        List<Runnable> toRun = new ArrayList<>();
        while (true) {
            synchronized (this) {
                if (pendingCalls.isEmpty()) {
                    pendingCalls = null;
                    passThrough = true;
                    break;
                }
                List<Runnable> tmp = toRun;
                toRun = pendingCalls;
                pendingCalls = tmp;
            }
            for (Runnable runnable : toRun) {
                runnable.run();
            }
            toRun.clear();
        }
    }

    private void delayOrExecute(Runnable runnable) {
        synchronized (this) {
            if (!passThrough) {
                pendingCalls.add(runnable);
                return;
            }
        }
        runnable.run();
    }

}
