package ru.yandex.intranet.d.grpc.interceptors;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.grpc.Context;
import io.grpc.Contexts;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import net.devh.boot.grpc.common.util.InterceptorOrder;
import net.devh.boot.grpc.server.interceptor.GrpcGlobalServerInterceptor;
import org.springframework.core.annotation.Order;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import static ru.yandex.intranet.d.i18n.Locales.DEFAULT_LOCALE;
import static ru.yandex.intranet.d.i18n.Locales.SUPPORTED_LOCALES;

/**
 * GRPC locale interceptor.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcGlobalServerInterceptor
@Order(InterceptorOrder.ORDER_FIRST + 1)
public class LocaleInterceptor implements ServerInterceptor {

    public static final Context.Key<Locale> LOCALE_KEY = Context.key("locale");

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(ServerCall<ReqT, RespT> call,
                                                                 Metadata headers,
                                                                 ServerCallHandler<ReqT, RespT> next) {
        Locale locale = resolveSupportedLocale(headers);
        Context context = Context.current().withValue(LOCALE_KEY, locale);
        return Contexts.interceptCall(context, call, headers, next);
    }

    private Locale resolveSupportedLocale(Metadata headers) {
        List<Locale> requestLocales;
        try {
            requestLocales = getAcceptLanguageAsLocales(headers);
        } catch (Exception e) {
            return DEFAULT_LOCALE;
        }
        if (CollectionUtils.isEmpty(requestLocales)) {
            return DEFAULT_LOCALE;
        }
        Locale languageMatch = null;
        for (Locale locale : requestLocales) {
            if (SUPPORTED_LOCALES.contains(locale)) {
                if (languageMatch == null || languageMatch.getLanguage().equals(locale.getLanguage())) {
                    return locale;
                }
            } else if (languageMatch == null) {
                for (Locale candidate : SUPPORTED_LOCALES) {
                    if (!StringUtils.hasLength(candidate.getCountry())
                            && candidate.getLanguage().equals(locale.getLanguage())) {
                        languageMatch = candidate;
                        break;
                    }
                }
            }
        }
        if (languageMatch != null) {
            return languageMatch;
        }
        return DEFAULT_LOCALE;
    }

    private List<Locale> getAcceptLanguageAsLocales(Metadata headers) {
        List<Locale.LanguageRange> ranges = getAcceptLanguage(headers);
        if (ranges.isEmpty()) {
            return Collections.emptyList();
        }
        return ranges.stream()
                .map(range -> Locale.forLanguageTag(range.getRange()))
                .filter(locale -> StringUtils.hasText(locale.getDisplayName()))
                .collect(Collectors.toList());
    }

    private List<Locale.LanguageRange> getAcceptLanguage(Metadata headers) {
        String value = headers.get(Metadata.Key.of("Accept-Language", Metadata.ASCII_STRING_MARSHALLER));
        return (StringUtils.hasText(value) ? Locale.LanguageRange.parse(value) : Collections.emptyList());
    }


}
