package ru.yandex.intranet.d.grpc.services;

import java.util.Locale;
import java.util.Optional;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.backend.service.proto.AccountKey;
import ru.yandex.intranet.d.backend.service.proto.AccountName;
import ru.yandex.intranet.d.backend.service.proto.AccountOperationStatus;
import ru.yandex.intranet.d.backend.service.proto.AccountsPageToken;
import ru.yandex.intranet.d.backend.service.proto.AccountsServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.CreateAccountRequest;
import ru.yandex.intranet.d.backend.service.proto.CreateAccountResponse;
import ru.yandex.intranet.d.backend.service.proto.GetFolderAccountRequest;
import ru.yandex.intranet.d.backend.service.proto.ListAccountsByFolderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListAccountsByFolderResponse;
import ru.yandex.intranet.d.backend.service.proto.ListProviderAccountsByFolderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListProviderAccountsByFolderResponse;
import ru.yandex.intranet.d.backend.service.proto.ProviderAccount;
import ru.yandex.intranet.d.backend.service.proto.ProviderAccountsSpace;
import ru.yandex.intranet.d.backend.service.proto.PutAccountRequest;
import ru.yandex.intranet.d.backend.service.proto.PutAccountResponse;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.grpc.GrpcIdempotency;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.services.accounts.AccountLogicService;
import ru.yandex.intranet.d.services.accounts.AccountService;
import ru.yandex.intranet.d.services.accounts.AccountsReadService;
import ru.yandex.intranet.d.services.accounts.model.AccountOperationResult;
import ru.yandex.intranet.d.services.accounts.model.AccountOperationResultStatus;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeDto;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeInputDto;
import ru.yandex.intranet.d.web.model.accounts.CreateAccountDto;
import ru.yandex.intranet.d.web.model.accounts.PutAccountDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC accounts service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcAccountsServiceImpl extends AccountsServiceGrpc.AccountsServiceImplBase {

    private final AccountsReadService accountsReadService;
    private final AccountService accountService;
    private final AccountLogicService accountLogicService;
    private final MessageSource messages;

    public GrpcAccountsServiceImpl(AccountsReadService accountsReadService,
                                   AccountService accountService,
                                   AccountLogicService accountLogicService,
                                   @Qualifier("messageSource") MessageSource messages) {
        this.accountsReadService = accountsReadService;
        this.accountService = accountService;
        this.accountLogicService = accountLogicService;
        this.messages = messages;
    }

    @Override
    public void listAccountsByFolder(ListAccountsByFolderRequest request,
                                     StreamObserver<ListAccountsByFolderResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountsReadService.getForFolder(reqParam.getFolderId(), toPageRequest(reqParam),
                                currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPageByFolder(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void listProviderAccountsByFolder(ListProviderAccountsByFolderRequest request,
                                             StreamObserver<ListProviderAccountsByFolderResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountsReadService.getForProvider(reqParam.getFolderId(),
                                reqParam.getProviderId(), toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPageByProvider(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getFolderAccount(GetFolderAccountRequest request, StreamObserver<ProviderAccount> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountsReadService.getById(reqParam.getFolderId(), reqParam.getAccountId(),
                                currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toAccount(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createAccount(CreateAccountRequest request, StreamObserver<CreateAccountResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Optional<String> idempotencyKey = GrpcIdempotency.idempotencyKey();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountLogicService.createAccountMono(reqParam.getFolderId(),
                                toCreateAccount(request), currentUser, locale, idempotencyKey.orElse(null))
                        .flatMap(resp -> resp.match(u -> Mono.just(toCreateAccountOperation(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void putAccount(PutAccountRequest request, StreamObserver<PutAccountResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Optional<String> idempotencyKey = GrpcIdempotency.idempotencyKey();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountLogicService.putAccountMono(reqParam.getFolderId(),
                                reqParam.getAccountId(), toPutAccount(request), currentUser, locale,
                                idempotencyKey.orElse(null))
                        .flatMap(resp -> resp.match(u -> Mono.just(toPutAccountOperation(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private ProviderAccount toAccount(AccountModel account) {
        ProviderAccount.Builder builder = ProviderAccount.newBuilder();
        builder.setId(account.getId());
        builder.setExternalId(account.getOuterAccountIdInProvider());
        account.getOuterAccountKeyInProvider().ifPresent(key -> builder.setExternalKey(AccountKey.newBuilder()
                .setValue(key).build()));
        builder.setVersion(account.getVersion());
        builder.setFolderId(account.getFolderId());
        builder.setProviderId(account.getProviderId());
        account.getDisplayName().ifPresent(name -> builder.setDisplayName(AccountName.newBuilder()
                .setValue(name).build()));
        account.getAccountsSpacesId().ifPresent(id -> builder.setAccountsSpace(ProviderAccountsSpace.newBuilder()
                .setId(id).build()));
        builder.setFreeTier(account.isFreeTier());
        builder.setReserveType(AccountReserveTypeDto.toProto(account.getReserveType().orElse(null)));
        return builder.build();
    }

    private ListAccountsByFolderResponse toPageByFolder(Page<AccountModel> page) {
        ListAccountsByFolderResponse.Builder builder = ListAccountsByFolderResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(AccountsPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addAccounts(toAccount(e)));
        return builder.build();
    }

    private ListProviderAccountsByFolderResponse toPageByProvider(Page<AccountModel> page) {
        ListProviderAccountsByFolderResponse.Builder builder = ListProviderAccountsByFolderResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(AccountsPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addAccounts(toAccount(e)));
        return builder.build();
    }

    private PageRequest toPageRequest(ListAccountsByFolderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private PageRequest toPageRequest(ListProviderAccountsByFolderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private CreateAccountDto toCreateAccount(CreateAccountRequest request) {
        CreateAccountDto.Builder builder = CreateAccountDto.builder()
                .providerId(request.getProviderId())
                .freeTier(request.getFreeTier());
        if (request.hasAccountsSpace()) {
            builder.accountsSpaceId(request.getAccountsSpace().getId());
        }
        if (request.hasExternalKey()) {
            builder.externalKey(request.getExternalKey().getValue());
        }
        if (request.hasDisplayName()) {
            builder.displayName(request.getDisplayName().getValue());
        }
        builder.reserveType(AccountReserveTypeInputDto.fromProto(request.getReserveType()));
        return builder.build();
    }

    private PutAccountDto toPutAccount(PutAccountRequest request) {
        return new PutAccountDto(AccountReserveTypeInputDto.fromProto(request.getReserveType()), request.getVersion());
    }

    private CreateAccountResponse toCreateAccountOperation(AccountOperationResult operation) {
        CreateAccountResponse.Builder builder = CreateAccountResponse.newBuilder();
        builder.setOperationId(operation.getOperationId());
        builder.setOperationStatus(toOperationStatus(operation.getOperationStatus()));
        operation.getAccount().ifPresent(r -> builder.setAccount(toAccount(r)));
        return builder.build();
    }

    private PutAccountResponse toPutAccountOperation(AccountOperationResult operation) {
        PutAccountResponse.Builder builder = PutAccountResponse.newBuilder();
        builder.setOperationId(operation.getOperationId());
        builder.setOperationStatus(toOperationStatus(operation.getOperationStatus()));
        operation.getAccount().ifPresent(r -> builder.setAccount(toAccount(r)));
        return builder.build();
    }

    private AccountOperationStatus toOperationStatus(AccountOperationResultStatus status) {
        switch (status) {
            case SUCCESS:
                return AccountOperationStatus.ACCOUNT_OPERATION_SUCCESS;
            case IN_PROGRESS:
                return AccountOperationStatus.ACCOUNT_OPERATION_IN_PROGRESS;
            default:
                throw new IllegalArgumentException("Unsupported operation status: " + status);
        }
    }

}
