package ru.yandex.intranet.d.grpc.services;

import java.util.Locale;
import java.util.Map;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.AccountsSpace;
import ru.yandex.intranet.d.backend.service.proto.AccountsSpacesPageToken;
import ru.yandex.intranet.d.backend.service.proto.AccountsSpacesServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.GetAccountsSpaceRequest;
import ru.yandex.intranet.d.backend.service.proto.ListAccountsSpacesByProviderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListAccountsSpacesByProviderResponse;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.services.resources.AccountsSpacesService;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GrpcAccountsSpacesServiceImpl.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 19.01.2021
 */
@GrpcService
public class GrpcAccountsSpacesServiceImpl extends AccountsSpacesServiceGrpc.AccountsSpacesServiceImplBase {
    private final AccountsSpacesService accountsSpacesService;
    private final MessageSource messages;
    private final CommonProtoConverter converter;

    public GrpcAccountsSpacesServiceImpl(
            AccountsSpacesService accountsSpacesService,
            @Qualifier("messageSource") MessageSource messages,
            CommonProtoConverter converter
    ) {
        this.accountsSpacesService = accountsSpacesService;
        this.messages = messages;
        this.converter = converter;
    }

    @Override
    public void getAccountsSpace(GetAccountsSpaceRequest request, StreamObserver<AccountsSpace> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req.flatMap(reqParam ->
                accountsSpacesService.getByIdExpanded(
                        reqParam.getAccountsSpaceId(), reqParam.getProviderId(), currentUser, locale
                ).flatMap(resp -> resp.match(
                        a -> Mono.just(toProto(a.getAccountsSpaces(), a.getSegmentations(), a.getSegments(), locale)),
                        e -> Mono.error(Errors.toGrpcError(e, messages, locale))
                ))), messages);
    }

    @Override
    public void listAccountsSpacesByProvider(
            ListAccountsSpacesByProviderRequest request,
            StreamObserver<ListAccountsSpacesByProviderResponse> responseObserver
    ) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountsSpacesService.getPageExpanded(
                        reqParam.getProviderId(), toPageRequest(reqParam), currentUser, locale, false
                ).flatMap(resp -> resp.match(u -> Mono.just(toPageProto(u, locale)),
                        e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private ListAccountsSpacesByProviderResponse toPageProto(
            ExpandedAccountsSpaces<Page<AccountSpaceModel>> expandedPage, Locale locale
    ) {
        Page<AccountSpaceModel> page = expandedPage.getAccountsSpaces();
        Map<String, ResourceSegmentationModel> segmentations = expandedPage.getSegmentations();
        Map<String, ResourceSegmentModel> segments = expandedPage.getSegments();
        ListAccountsSpacesByProviderResponse.Builder builder = ListAccountsSpacesByProviderResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(AccountsSpacesPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(accountSpace -> builder.addAccountsSpaces(
                toProto(accountSpace, segmentations, segments, locale)));
        return builder.build();
    }

    private AccountsSpace toProto(
            AccountSpaceModel m,
            Map<String, ResourceSegmentationModel> segmentations,
            Map<String, ResourceSegmentModel> segments,
            Locale locale
    ) {
        AccountsSpace.Builder builder = AccountsSpace.newBuilder();
        builder
                .setAccountsSpaceId(m.getId())
                .setProviderId(m.getProviderId())
                .setVersion(m.getVersion())
                .setName(Locales.select(m.getNameEn(), m.getNameRu(), locale))
                .setDescription(Locales.select(m.getDescriptionEn(), m.getDescriptionRu(), locale))
                .setReadOnly(m.isReadOnly())
                .setSyncEnabled(m.isSyncEnabled());
        if (m.getOuterKeyInProvider() != null) {
            builder.setKey(m.getOuterKeyInProvider());
        }
        m.getSegments().forEach(segment -> builder.addSegments(converter.toSegment(
                segmentations.get(segment.getSegmentationId()),
                segments.get(segment.getSegmentId()),
                locale
        )));
        return builder.build();
    }

    private PageRequest toPageRequest(ListAccountsSpacesByProviderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }
}
