package ru.yandex.intranet.d.grpc.services

import com.google.protobuf.util.Timestamps
import io.grpc.stub.StreamObserver
import net.devh.boot.grpc.server.service.GrpcService
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.backend.service.proto.AggregateAmount
import ru.yandex.intranet.d.backend.service.proto.AggregateAmounts
import ru.yandex.intranet.d.backend.service.proto.AggregateUsage
import ru.yandex.intranet.d.backend.service.proto.AggregateUsageHistogram
import ru.yandex.intranet.d.backend.service.proto.AggregateUsageHistogramBin
import ru.yandex.intranet.d.backend.service.proto.AggregateUsageTimeSeries
import ru.yandex.intranet.d.backend.service.proto.AggregateUsageTimeSeriesPoint
import ru.yandex.intranet.d.backend.service.proto.AggregatesPageToken
import ru.yandex.intranet.d.backend.service.proto.AggregationQueriesServiceGrpc
import ru.yandex.intranet.d.backend.service.proto.FindSubtreeTotalAggregate
import ru.yandex.intranet.d.backend.service.proto.FindSubtreeTotalRequest
import ru.yandex.intranet.d.backend.service.proto.FindSubtreeTotalResponse
import ru.yandex.intranet.d.backend.service.proto.FloatingPointUsageValue
import ru.yandex.intranet.d.backend.service.proto.RankSubtreeAmountsAggregate
import ru.yandex.intranet.d.backend.service.proto.RankSubtreeAmountsRequest
import ru.yandex.intranet.d.backend.service.proto.RankSubtreeAmountsResponse
import ru.yandex.intranet.d.backend.service.proto.RankSubtreeSortingField
import ru.yandex.intranet.d.backend.service.proto.RankSubtreeSortingOrder
import ru.yandex.intranet.d.backend.service.proto.RankSubtreeSortingParams
import ru.yandex.intranet.d.grpc.Grpc
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.services.aggregates.QueryAggregatesService
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.ALLOCATED
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.BALANCE
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.PROVIDED
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.QUOTA
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.TRANSFERABLE
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.UNALLOCATED
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.UNDERUTILIZED_ESTIMATION
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.UNUSED_ESTIMATION
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingOrder.ASC
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingOrder.DESC
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingParamsDto
import ru.yandex.intranet.d.web.model.aggregation.api.AggregateUsageApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.FindSubtreeTotalApiRequestDto
import ru.yandex.intranet.d.web.model.aggregation.api.FindSubtreeTotalApiResponseDto
import ru.yandex.intranet.d.web.model.aggregation.api.RankSubtreeAmountsApiRequestDto
import ru.yandex.intranet.d.web.model.aggregation.api.RankSubtreeAmountsApiResponseDto
import ru.yandex.intranet.d.web.security.Auth

/**
 * GRPC aggregation queries service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
open class GrpcAggregationQueriesServiceImpl(
    private val queryAggregatesService: QueryAggregatesService,
    @Qualifier("messageSource") private val messages: MessageSource
): AggregationQueriesServiceGrpc.AggregationQueriesServiceImplBase() {

    override fun findSubtreeTotal(
        request: FindSubtreeTotalRequest,
        responseObserver: StreamObserver<FindSubtreeTotalResponse>
    ) {
        val currentUser = Auth.grpcUser()
        val locale = Locales.grpcLocale()
        Grpc.oneToOne(request, responseObserver,
            { req -> req.flatMap { reqParam -> queryAggregatesService.findSubtreeTotalApiMono(fromRequest(reqParam),
                currentUser, locale).flatMap { resp -> resp.match({ folderModel -> Mono.just(toResponse(folderModel)) })
                    { errorCollection -> Mono.error(Errors.toGrpcError(errorCollection, messages, locale)) } } }
            }, messages)
    }

    override fun rankSubtreeAmounts(
        request: RankSubtreeAmountsRequest,
        responseObserver: StreamObserver<RankSubtreeAmountsResponse>
    ) {
        val currentUser = Auth.grpcUser()
        val locale = Locales.grpcLocale()
        Grpc.oneToOne(request, responseObserver,
            { req -> req.flatMap { reqParam -> queryAggregatesService.rankSubtreeAmountsApiMono(fromRequest(reqParam),
                currentUser, locale).flatMap { resp -> resp.match({ folderModel -> Mono.just(toResponse(folderModel)) })
                    { errorCollection -> Mono.error(Errors.toGrpcError(errorCollection, messages, locale)) } } }
            }, messages)
    }

    fun fromRequest(request: FindSubtreeTotalRequest) = FindSubtreeTotalApiRequestDto(
        rootServiceId = request.rootServiceId,
        resourceId = request.resourceId,
        providerId = request.providerId,
        includeUsage = request.includeUsage,
        includeUsageRaw = request.includeUsageRaw
    )

    fun toResponse(response: FindSubtreeTotalApiResponseDto): FindSubtreeTotalResponse {
        val builder = FindSubtreeTotalResponse.newBuilder()
        val amountsBuilder = AggregateAmounts.newBuilder()
        if (response.aggregate.amounts.quota != null) {
            amountsBuilder.setQuota(AggregateAmount.newBuilder()
                .setAmount(response.aggregate.amounts.quota.amount)
                .setUnitKey(response.aggregate.amounts.quota.unitKey)
                .build())
        }
        if (response.aggregate.amounts.balance != null) {
            amountsBuilder.setBalance(AggregateAmount.newBuilder()
                .setAmount(response.aggregate.amounts.balance.amount)
                .setUnitKey(response.aggregate.amounts.balance.unitKey)
                .build())
        }
        if (response.aggregate.amounts.provided != null) {
            amountsBuilder.setProvided(AggregateAmount.newBuilder()
                .setAmount(response.aggregate.amounts.provided.amount)
                .setUnitKey(response.aggregate.amounts.provided.unitKey)
                .build())
        }
        if (response.aggregate.amounts.allocated != null) {
            amountsBuilder.setAllocated(AggregateAmount.newBuilder()
                .setAmount(response.aggregate.amounts.allocated.amount)
                .setUnitKey(response.aggregate.amounts.allocated.unitKey)
                .build())
        }
        if (response.aggregate.amounts.unallocated != null) {
            amountsBuilder.unallocated = AggregateAmount.newBuilder()
                .setAmount(response.aggregate.amounts.unallocated.amount)
                .setUnitKey(response.aggregate.amounts.unallocated.unitKey)
                .build()
        }
        if (response.aggregate.amounts.transferable != null) {
            amountsBuilder.transferable = AggregateAmount.newBuilder()
                .setAmount(response.aggregate.amounts.transferable.amount)
                .setUnitKey(response.aggregate.amounts.transferable.unitKey)
                .build()
        }
        val aggregateBuilder = FindSubtreeTotalAggregate.newBuilder()
            .setAmounts(amountsBuilder.build())
        if (response.aggregate.lastUpdate != null) {
            aggregateBuilder.setLastUpdate(Timestamps.fromMillis(response.aggregate.lastUpdate.toEpochMilli()))
        }
        if (response.aggregate.usage != null) {
            aggregateBuilder.setUsage(toUsage(response.aggregate.usage))
        }
        builder.setAggregate(aggregateBuilder.build())
        return builder.build()
    }

    fun toUsage(usage: AggregateUsageApiDto): AggregateUsage {
        val usageBuilder = AggregateUsage.newBuilder()
        if (usage.value != null) {
            usageBuilder.setValue(AggregateAmount.newBuilder()
                .setAmount(usage.value.amount)
                .setUnitKey(usage.value.unitKey)
                .build())
        }
        if (usage.average != null) {
            usageBuilder.setAverage(AggregateAmount.newBuilder()
                .setAmount(usage.average.amount)
                .setUnitKey(usage.average.unitKey)
                .build())
        }
        if (usage.min != null) {
            usageBuilder.setMin(AggregateAmount.newBuilder()
                .setAmount(usage.min.amount)
                .setUnitKey(usage.min.unitKey)
                .build())
        }
        if (usage.max != null) {
            usageBuilder.setMax(AggregateAmount.newBuilder()
                .setAmount(usage.max.amount)
                .setUnitKey(usage.max.unitKey)
                .build())
        }
        if (usage.median != null) {
            usageBuilder.setMedian(AggregateAmount.newBuilder()
                .setAmount(usage.median.amount)
                .setUnitKey(usage.median.unitKey)
                .build())
        }
        if (usage.stdev != null) {
            usageBuilder.setStdev(AggregateAmount.newBuilder()
                .setAmount(usage.stdev.amount)
                .setUnitKey(usage.stdev.unitKey)
                .build())
        }
        if (usage.relativeUsage != null) {
            usageBuilder.setRelativeUsage(FloatingPointUsageValue.newBuilder()
                .setValue(usage.relativeUsage)
                .build())
        }
        if (usage.kv != null) {
            usageBuilder.setKv(FloatingPointUsageValue.newBuilder()
                .setValue(usage.kv)
                .build())
        }
        if (usage.histogram != null) {
            val histogramBuilder = AggregateUsageHistogram.newBuilder()
            usage.histogram.forEach { bin ->
                histogramBuilder.addBins(AggregateUsageHistogramBin.newBuilder()
                    .setFrom(AggregateAmount.newBuilder()
                        .setAmount(bin.from.amount)
                        .setUnitKey(bin.from.unitKey)
                        .build())
                    .setTo(AggregateAmount.newBuilder()
                        .setAmount(bin.to.amount)
                        .setUnitKey(bin.to.unitKey)
                        .build())
                    .setValue(bin.value)
                    .build())
            }
            usageBuilder.setHistogram(histogramBuilder.build())
        }
        if (usage.timeSeries != null) {
            val timeSeriesBuilder = AggregateUsageTimeSeries.newBuilder()
            usage.timeSeries.forEach { point ->
                timeSeriesBuilder.addPoints(AggregateUsageTimeSeriesPoint.newBuilder()
                    .setX(Timestamps.fromMillis(point.x.toEpochMilli()))
                    .setY(AggregateAmount.newBuilder()
                        .setAmount(point.y.amount)
                        .setUnitKey(point.y.unitKey)
                        .build())
                    .build())
            }
            usageBuilder.setTimeSeries(timeSeriesBuilder.build())
        }
        if (usage.unusedEstimation != null) {
            usageBuilder.setUnusedEstimation(AggregateAmount.newBuilder()
                .setAmount(usage.unusedEstimation.amount)
                .setUnitKey(usage.unusedEstimation.unitKey)
                .build())
        }
        if (usage.underutilizedEstimation != null) {
            usageBuilder.setUnderutilizedEstimation(AggregateAmount.newBuilder()
                .setAmount(usage.underutilizedEstimation.amount)
                .setUnitKey(usage.underutilizedEstimation.unitKey)
                .build())
        }
        return usageBuilder.build()
    }

    fun fromRequest(request: RankSubtreeAmountsRequest) = RankSubtreeAmountsApiRequestDto(
        rootServiceId = request.rootServiceId,
        resourceId = request.resourceId,
        providerId = request.providerId,
        from = if (request.hasFrom()) { request.from.token } else { null },
        limit = if (request.hasLimit()) { request.limit.limit } else { null },
        sortingParams = fromSortingParams(request.sortingParams),
        includeUsage = request.includeUsage,
        includeUsageRaw = request.includeUsageRaw
    )

    fun toResponse(response: RankSubtreeAmountsApiResponseDto): RankSubtreeAmountsResponse {
        val builder = RankSubtreeAmountsResponse.newBuilder()
        response.aggregates.forEach {
            val aggregateBuilder = RankSubtreeAmountsAggregate.newBuilder()
            aggregateBuilder.setServiceId(it.serviceId)
            if (it.lastUpdate != null) {
                aggregateBuilder.setLastUpdate(Timestamps.fromMillis(it.lastUpdate.toEpochMilli()))
            }
            val amountsBuilder = AggregateAmounts.newBuilder()
            if (it.amounts.quota != null) {
                amountsBuilder.setQuota(AggregateAmount.newBuilder()
                    .setAmount(it.amounts.quota.amount)
                    .setUnitKey(it.amounts.quota.unitKey)
                    .build())
            }
            if (it.amounts.balance != null) {
                amountsBuilder.setBalance(AggregateAmount.newBuilder()
                    .setAmount(it.amounts.balance.amount)
                    .setUnitKey(it.amounts.balance.unitKey)
                    .build())
            }
            if (it.amounts.provided != null) {
                amountsBuilder.setProvided(AggregateAmount.newBuilder()
                    .setAmount(it.amounts.provided.amount)
                    .setUnitKey(it.amounts.provided.unitKey)
                    .build())
            }
            if (it.amounts.allocated != null) {
                amountsBuilder.setAllocated(AggregateAmount.newBuilder()
                    .setAmount(it.amounts.allocated.amount)
                    .setUnitKey(it.amounts.allocated.unitKey)
                    .build())
            }
            if (it.amounts.unallocated != null) {
                amountsBuilder.unallocated = AggregateAmount.newBuilder()
                    .setAmount(it.amounts.unallocated.amount)
                    .setUnitKey(it.amounts.unallocated.unitKey)
                    .build()
            }
            if (it.amounts.transferable != null) {
                amountsBuilder.transferable = AggregateAmount.newBuilder()
                    .setAmount(it.amounts.transferable.amount)
                    .setUnitKey(it.amounts.transferable.unitKey)
                    .build()
            }
            aggregateBuilder.setAmounts(amountsBuilder.build())
            if (it.usage != null) {
                aggregateBuilder.setUsage(toUsage(it.usage))
            }
            builder.addAggregates(aggregateBuilder.build())
        }
        if (response.nextPageToken != null) {
            builder.setNext(AggregatesPageToken.newBuilder().setToken(response.nextPageToken).build())
        }
        return builder.build()
    }

    private fun fromSortingParams(sortingParams: RankSubtreeSortingParams): RankSubtreeSortingParamsDto {
        val field = when (sortingParams.sortingField) {
            RankSubtreeSortingField.ALLOCATED -> ALLOCATED
            RankSubtreeSortingField.BALANCE -> BALANCE
            RankSubtreeSortingField.TRANSFERABLE -> TRANSFERABLE
            RankSubtreeSortingField.PROVIDED -> PROVIDED
            RankSubtreeSortingField.QUOTA -> QUOTA
            RankSubtreeSortingField.UNALLOCATED -> UNALLOCATED
            RankSubtreeSortingField.UNUSED_ESTIMATION -> UNUSED_ESTIMATION
            RankSubtreeSortingField.UNDERUTILIZED_ESTIMATION -> UNDERUTILIZED_ESTIMATION
            else -> throw IllegalArgumentException("Unsupported sorting field: ${sortingParams.sortingField}")
        }
        val order = when (sortingParams.sortingOrder) {
            RankSubtreeSortingOrder.DESC -> DESC
            RankSubtreeSortingOrder.ASC -> ASC
            else -> throw IllegalArgumentException("Unsupported sorting order: ${sortingParams.sortingOrder}")
        }
        return RankSubtreeSortingParamsDto(field, order)
    }
}
