package ru.yandex.intranet.d.grpc.services;

import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.Folder;
import ru.yandex.intranet.d.backend.service.proto.FolderServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.FolderType;
import ru.yandex.intranet.d.backend.service.proto.FoldersPageToken;
import ru.yandex.intranet.d.backend.service.proto.GetFolderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListFoldersByIdsRequest;
import ru.yandex.intranet.d.backend.service.proto.ListFoldersByIdsResponse;
import ru.yandex.intranet.d.backend.service.proto.ListFoldersByServiceRequest;
import ru.yandex.intranet.d.backend.service.proto.ListFoldersByServiceResponse;
import ru.yandex.intranet.d.backend.service.proto.ListFoldersRequest;
import ru.yandex.intranet.d.backend.service.proto.ListFoldersResponse;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC folder service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcFolderServiceImpl extends FolderServiceGrpc.FolderServiceImplBase {

    private final MessageSource messages;
    private final FolderService folderService;

    public GrpcFolderServiceImpl(
            @Qualifier("messageSource") MessageSource messages,
            FolderService folderService
    ) {
        this.messages = messages;
        this.folderService = folderService;
    }

    @Override
    public void getFolder(GetFolderRequest request, StreamObserver<Folder> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(
                request,
                responseObserver,
                req -> req.flatMap(reqParam -> folderService.getFolder(reqParam.getFolderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(
                                folderModel -> Mono.just(toFolderProto(folderModel)),
                                errorCollection -> Mono.error(Errors.toGrpcError(errorCollection, messages, locale))
                        ))
                ), messages
        );
    }

    @Override
    public void listFoldersByService(
            ListFoldersByServiceRequest request,
            StreamObserver<ListFoldersByServiceResponse> responseObserver
    ) {
        PageRequest pageRequest = new PageRequest(
                request.hasPageToken() ? request.getPageToken().getToken() : null,
                request.hasLimit() ? request.getLimit().getLimit() : null
        );
        long serviceId = request.getServiceId();
        boolean includeDeleted = request.getIncludeDeleted();
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();

        Grpc.oneToOne(
                request,
                responseObserver,
                req -> req.flatMap(reqParam ->
                        folderService.listFoldersByService(
                                serviceId,
                                includeDeleted,
                                pageRequest,
                                currentUser,
                                locale
                        ).flatMap(resp -> resp.match(
                                folders -> Mono.just(toListFoldersByServiceResponse(folders)),
                                errors -> Mono.error(Errors.toGrpcError(errors, messages, locale))
                        ))
                ), messages
        );
    }

    @Override
    public void listFolders(ListFoldersRequest request, StreamObserver<ListFoldersResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> folderService.listFolders(reqParam.getIncludeDeleted(),
                        toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPage(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void listFoldersByIds(ListFoldersByIdsRequest request,
                                 StreamObserver<ListFoldersByIdsResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> folderService.getFolders(reqParam.getFolderIdList(),
                        reqParam.getIncludeDeleted(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toFolders(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private ListFoldersByServiceResponse toListFoldersByServiceResponse(Page<FolderModel> foldersPage) {
        ListFoldersByServiceResponse.Builder builder = ListFoldersByServiceResponse.newBuilder()
                .addAllFolders(toFolderProto(foldersPage));
        foldersPage.getContinuationToken().ifPresent(token ->
                builder.setNextPageToken(FoldersPageToken.newBuilder().setToken(token))
        );
        return builder.build();
    }

    List<Folder> toFolderProto(Page<FolderModel> foldersPage) {
        return foldersPage.getItems().stream().map(this::toFolderProto).collect(Collectors.toList());
    }

    private Folder toFolderProto(FolderModel in) {
        return Folder.newBuilder()
                .setFolderId(in.getId())
                .setServiceId(in.getServiceId())
                .setDisplayName(in.getDisplayName())
                .setDeleted(in.isDeleted())
                .setDescription("")
                .setVersion(0L)
                .setFolderType(convert(in.getFolderType()))
                .build();
    }

    private FolderType convert(ru.yandex.intranet.d.model.folders.FolderType folderType) {
        switch (folderType) {
            case COMMON:
                return FolderType.COMMON;
            case COMMON_DEFAULT_FOR_SERVICE:
                return FolderType.COMMON_DEFAULT_FOR_SERVICE;
            case PROVIDER_RESERVE:
                return FolderType.PROVIDER_RESERVE;
            default:
                throw new UnsupportedOperationException("Unsupported FolderType value: " + folderType);
        }
    }

    private PageRequest toPageRequest(ListFoldersRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private ListFoldersResponse toPage(Page<FolderModel> page) {
        ListFoldersResponse.Builder builder = ListFoldersResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(FoldersPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addFolders(toFolderProto(e)));
        return builder.build();
    }

    private ListFoldersByIdsResponse toFolders(List<FolderModel> folders) {
        ListFoldersByIdsResponse.Builder builder = ListFoldersByIdsResponse.newBuilder();
        folders.forEach(e -> builder.addFolders(toFolderProto(e)));
        return builder.build();
    }

}
