package ru.yandex.intranet.d.grpc.services

import com.google.protobuf.util.Timestamps
import io.grpc.stub.StreamObserver
import net.devh.boot.grpc.server.service.GrpcService
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.backend.service.proto.LoanActionSubject
import ru.yandex.intranet.d.backend.service.proto.LoanActionSubjects
import ru.yandex.intranet.d.backend.service.proto.LoanAmount
import ru.yandex.intranet.d.backend.service.proto.LoanAmounts
import ru.yandex.intranet.d.backend.service.proto.LoanSubject
import ru.yandex.intranet.d.backend.service.proto.LoansHistory
import ru.yandex.intranet.d.backend.service.proto.LoansHistoryFields
import ru.yandex.intranet.d.backend.service.proto.LoansServiceGrpc
import ru.yandex.intranet.d.backend.service.proto.LocalDate
import ru.yandex.intranet.d.grpc.Grpc
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.model.loans.LoanEventType
import ru.yandex.intranet.d.model.loans.LoanStatus
import ru.yandex.intranet.d.model.loans.LoanSubjectType
import ru.yandex.intranet.d.model.loans.LoanType
import ru.yandex.intranet.d.services.loans.LoansService
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.loans.LoanActionSubjectDto
import ru.yandex.intranet.d.web.model.loans.LoanActionSubjectsDto
import ru.yandex.intranet.d.web.model.loans.LoanAmountDto
import ru.yandex.intranet.d.web.model.loans.LoanAmountsDto
import ru.yandex.intranet.d.web.model.loans.LoanDirection
import ru.yandex.intranet.d.web.model.loans.LoanDueDateDto
import ru.yandex.intranet.d.web.model.loans.LoanSubjectDto
import ru.yandex.intranet.d.web.model.loans.LoansHistoryFieldsDto
import ru.yandex.intranet.d.web.model.loans.api.ApiGetLoansHistoryResponseDto
import ru.yandex.intranet.d.web.model.loans.api.ApiLoanDto
import ru.yandex.intranet.d.web.model.loans.api.ApiLoansHistoryDto
import ru.yandex.intranet.d.web.model.loans.api.ApiSearchLoansRequestDto
import ru.yandex.intranet.d.web.model.loans.api.ApiSearchLoansResponseDto
import ru.yandex.intranet.d.web.security.Auth

typealias GetLoansHistoryRequestProto = ru.yandex.intranet.d.backend.service.proto.GetLoansHistoryRequest
typealias GetLoansHistoryResponseProto = ru.yandex.intranet.d.backend.service.proto.GetLoansHistoryResponse
typealias SearchLoansRequestProto = ru.yandex.intranet.d.backend.service.proto.SearchLoansRequest
typealias SearchLoansResponseProto = ru.yandex.intranet.d.backend.service.proto.SearchLoansResponse
typealias LoanProto = ru.yandex.intranet.d.backend.service.proto.Loan
typealias LoanDirectionProto = ru.yandex.intranet.d.backend.service.proto.LoanDirection
typealias LoanStatusProto = ru.yandex.intranet.d.backend.service.proto.LoanStatus
typealias LoanDueDateProto = ru.yandex.intranet.d.backend.service.proto.LoanDueDate
typealias LoanTypeProto = ru.yandex.intranet.d.backend.service.proto.LoanType
typealias LoanActionSubjectProto = LoanActionSubject
typealias LoanActionSubjectsProto = LoanActionSubjects
typealias LoanSubjectProto = LoanSubject
typealias LoanSubjectTypeProto = ru.yandex.intranet.d.backend.service.proto.LoanSubjectType
typealias LoanAmountsProto = LoanAmounts
typealias LoanAmountProto = LoanAmount
typealias LoansHistoryProto = LoansHistory
typealias LoanEventTypeProto = ru.yandex.intranet.d.backend.service.proto.LoanEventType
typealias LoansHistoryFieldsProto = LoansHistoryFields

@GrpcService
class GrpcLoansServiceImpl(
    private val loansService: LoansService,
    @Qualifier("messageSource") private val messages: MessageSource
) : LoansServiceGrpc.LoansServiceImplBase() {

    override fun searchLoans(
        request: SearchLoansRequestProto,
        responseObserver: StreamObserver<SearchLoansResponseProto>
    ) {
        val currentUser = Auth.grpcUser()
        val locale = Locales.grpcLocale()
        Grpc.oneToOne(
            request, responseObserver,
            { req ->
                req.flatMap { reqParam ->
                    loansService.searchLoansApiMono(reqParam.toDto(), currentUser, locale)
                        .flatMap { resp ->
                            resp.match(
                                onSuccess = { r -> Mono.just(r.toProto()) },
                                onFailure = { errorCollection ->
                                    Mono.error(Errors.toGrpcError(errorCollection, messages, locale))
                                }
                            )
                        }
                }
            },
            messages
        )
    }

    override fun getLoansHistory(
        request: GetLoansHistoryRequestProto,
        responseObserver: StreamObserver<GetLoansHistoryResponseProto>
    ) {
        val currentUser = Auth.grpcUser()
        val locale = Locales.grpcLocale()
        Grpc.oneToOne(
            request, responseObserver,
            { req ->
                req.flatMap { reqParam ->
                    loansService.getLoansHistoryApiMono(
                        reqParam.loanId,
                        reqParam.from.ifBlank { null },
                        reqParam.limit.takeIf { it != 0 },
                        currentUser, locale
                    ).flatMap { resp ->
                        resp.match(
                            onSuccess = { r -> Mono.just(r.toProto()) },
                            onFailure = { errorCollection ->
                                Mono.error(Errors.toGrpcError(errorCollection, messages, locale))
                            }
                        )
                    }
                }
            },
            messages
        )
    }

    private fun SearchLoansRequestProto.toDto() = ApiSearchLoansRequestDto(
        serviceId = serviceId,
        status = loanStatus.toDto(),
        direction = loanDirection.toDto(),
        from = from.ifBlank { null },
        limit = limit.takeIf { it != 0 }
    )

    private fun LoanStatusProto.toDto(): LoanStatus? =
        when (this) {
            LoanStatusProto.PENDING_LOAN -> LoanStatus.PENDING
            LoanStatusProto.SETTLED_LOAN -> LoanStatus.SETTLED
            LoanStatusProto.ANY, LoanStatusProto.UNRECOGNIZED -> null
        }

    private fun LoanStatus.toProto(): LoanStatusProto =
        when (this) {
            LoanStatus.PENDING -> LoanStatusProto.PENDING_LOAN
            LoanStatus.SETTLED -> LoanStatusProto.SETTLED_LOAN
        }

    private fun LoanDirectionProto.toDto(): LoanDirection? =
        when (this) {
            LoanDirectionProto.IN -> LoanDirection.IN
            LoanDirectionProto.OUT -> LoanDirection.OUT
            LoanDirectionProto.UNRECOGNIZED -> null
        }

    private fun ApiSearchLoansResponseDto.toProto(): SearchLoansResponseProto {
        val builder = SearchLoansResponseProto.newBuilder()
        loans.forEach { builder.addLoans(it.toProto()) }
        if (continuationToken != null) {
            builder.continuationToken = continuationToken
        }
        return builder.build()
    }

    private fun LoanDueDateDto.toProto(): LoanDueDateProto {
        val builder = LoanDueDateProto.newBuilder()
        val localDateBuilder = LocalDate.newBuilder()
        localDateBuilder.day = localDate.dayOfMonth
        localDateBuilder.month = localDate.month.ordinal
        localDateBuilder.year = localDate.year
        builder.localDate = localDateBuilder.build()
        builder.zoneId = timeZone.id
        return builder.build()
    }

    private fun LoanType.toProto(): LoanTypeProto = when (this) {
        LoanType.PROVIDER_RESERVE-> LoanTypeProto.PROVIDER_RESERVE_TYPE
        LoanType.PROVIDER_RESERVE_OVER_COMMIT -> LoanTypeProto.PROVIDER_RESERVE_OVER_COMMIT
        LoanType.SERVICE_RESERVE -> LoanTypeProto.SERVICE_RESERVE
        LoanType.PEER_TO_PEER -> LoanTypeProto.PEER_TO_PEER
    }

    private fun LoanActionSubjectDto.toProto(): LoanActionSubjectProto {
        val builder = LoanActionSubjectProto.newBuilder()
        if (user != null) {
            builder.user = user
        }
        if (provider != null) {
            builder.provider = provider
        }
        return builder.build()
    }

    private fun LoanActionSubjectsDto.toProto(): LoanActionSubjectsProto {
        val builder = LoanActionSubjectsProto.newBuilder()
        subjects.forEach { builder.addSubjects(it.toProto()) }
        return builder.build()
    }

    private fun LoanSubjectType.toProto(): LoanSubjectTypeProto = when (this) {
        LoanSubjectType.PROVIDER_RESERVE_ACCOUNT -> LoanSubjectTypeProto.PROVIDER_RESERVE_ACCOUNT
        LoanSubjectType.PROVIDER_RESERVE_OVER_COMMIT -> LoanSubjectTypeProto.PROVIDER_RESERVE_OVER_COMMIT_TYPE
        LoanSubjectType.SERVICE_RESERVE_ACCOUNT -> LoanSubjectTypeProto.SERVICE_RESERVE_ACCOUNT
        LoanSubjectType.ACCOUNT -> LoanSubjectTypeProto.ACCOUNT
        LoanSubjectType.SERVICE -> LoanSubjectTypeProto.SERVICE
    }

    private fun LoanSubjectDto.toProto(): LoanSubjectProto {
        val builder = LoanSubjectProto.newBuilder()
        builder.type = type.toProto()
        builder.service = service
        if (reserveService != null) {
            builder.reserveService = reserveService
        }
        if (account != null) {
            builder.account = account
        }
        if (folder != null) {
            builder.folder = folder
        }
        if (provider != null) {
            builder.provider = provider
        }
        return builder.build()
    }

    private fun LoanAmountsDto.toProto(): LoanAmountsProto {
        val builder = LoanAmountsProto.newBuilder()
        amounts.forEach { builder.addAmounts(it.toProto()) }
        return builder.build()
    }

    private fun LoanAmountDto.toProto(): LoanAmountProto {
        val builder = LoanAmountProto.newBuilder()
        builder.resource = resource
        builder.amount = amount.toString()
        return builder.build()
    }

    private fun ApiLoanDto.toProto(): LoanProto {
        val builder = LoanProto.newBuilder()
        builder.id = id
        builder.status = status.toProto()
        builder.type = type.toProto()
        builder.createdAt = Timestamps.fromMillis(createdAt.toEpochMilli())
        builder.dueAt = dueAt.toProto()
        if (settledAt != null) {
            builder.settledAt = Timestamps.fromMillis(settledAt.toEpochMilli())
        }
        if (updatedAt != null) {
            builder.updatedAt = Timestamps.fromMillis(updatedAt.toEpochMilli())
        }
        builder.version = version
        builder.requestedBy = requestedBy.toProto()
        builder.requestApprovedBy = requestApprovedBy.toProto()
        builder.borrowTransferRequestId = borrowTransferRequestId
        builder.borrowedFrom = borrowedFrom.toProto()
        builder.borrowedTo = borrowedTo.toProto()
        builder.payOffFrom = payOffFrom.toProto()
        builder.borrowedTo = borrowedTo.toProto()
        builder.borrowedAmounts = borrowedAmounts.toProto()
        builder.payOffAmounts = payOffAmounts.toProto()
        builder.dueAmounts = dueAmounts.toProto()
        return builder.build()
    }

    private fun ApiGetLoansHistoryResponseDto.toProto(): GetLoansHistoryResponseProto {
        val builder = GetLoansHistoryResponseProto.newBuilder()
        events.forEach { builder.addEvents(it.toProto()) }
        if (continuationToken != null) {
            builder.continuationToken = continuationToken
        }
        return builder.build()
    }

    private fun ApiLoansHistoryDto.toProto(): LoansHistoryProto {
        val builder = LoansHistoryProto.newBuilder()
        builder.historyId = historyId
        builder.loanId = loanId
        builder.eventTimestamp = Timestamps.fromMillis(eventTimestamp.toEpochMilli())
        builder.eventAuthor = eventAuthor.toProto()
        if (eventApprovedBy != null) {
            builder.eventApprovedBy = eventApprovedBy.toProto()
        }
        builder.eventType = eventType.toProto()
        builder.transferRequestId = transferRequestId
        if (oldFields != null) {
            builder.oldFields = oldFields.toProto()
        }
        if (newFields != null) {
            builder.newFields = newFields.toProto()
        }
        return builder.build()
    }

    private fun LoanEventType.toProto(): LoanEventTypeProto = when (this) {
        LoanEventType.LOAN_CREATED -> LoanEventTypeProto.LOAN_CREATED
        LoanEventType.LOAN_UPDATED -> LoanEventTypeProto.LOAN_UPDATED
        LoanEventType.LOAN_PAY_OFF -> LoanEventTypeProto.LOAN_PAY_OFF
        LoanEventType.LOAN_SETTLED -> LoanEventTypeProto.LOAN_SETTLED
    }

    private fun LoansHistoryFieldsDto.toProto(): LoansHistoryFieldsProto {
        val builder = LoansHistoryFieldsProto.newBuilder()
        builder.version = version
        if (status != null) {
            builder.status = status.toProto()
        }
        if (dueAt != null) {
            builder.dueAt = dueAt.toProto()
        }
        if (payOffFrom != null) {
            builder.payOffFrom = payOffFrom.toProto()
        }
        if (payOffTo != null) {
            builder.payOffTo = payOffTo.toProto()
        }
        if (borrowedAmounts != null) {
            builder.borrowedAmounts = borrowedAmounts.toProto()
        }
        if (payOffAmounts != null) {
            builder.payOffAmounts = payOffAmounts.toProto()
        }
        if (dueAmounts != null) {
            builder.dueAmounts = dueAmounts.toProto()
        }
        return builder.build()
    }
}
