package ru.yandex.intranet.d.grpc.services;

import java.util.Locale;

import com.google.protobuf.util.Timestamps;
import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.GetOperationStateRequest;
import ru.yandex.intranet.d.backend.service.proto.OperationErrorDetails;
import ru.yandex.intranet.d.backend.service.proto.OperationProviderAccountsSpace;
import ru.yandex.intranet.d.backend.service.proto.OperationState;
import ru.yandex.intranet.d.backend.service.proto.OperationStatus;
import ru.yandex.intranet.d.backend.service.proto.OperationsServiceGrpc;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.services.operations.OperationsService;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC operations service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcOperationsServiceImpl extends OperationsServiceGrpc.OperationsServiceImplBase {

    private final OperationsService operationsService;
    private final MessageSource messages;

    public GrpcOperationsServiceImpl(OperationsService operationsService,
                                     @Qualifier("messageSource") MessageSource messages) {
        this.operationsService = operationsService;
        this.messages = messages;
    }

    @Override
    public void getOperationState(GetOperationStateRequest request, StreamObserver<OperationState> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> operationsService.getById(reqParam.getOperationId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toOperation(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private OperationState toOperation(AccountsQuotasOperationsModel operation, Locale locale) {
        OperationState.Builder builder = OperationState.newBuilder();
        builder.setId(operation.getOperationId());
        builder.setProviderId(operation.getProviderId());
        builder.setStatus(toStatus(operation));
        builder.setCreatedAt(Timestamps.fromMillis(operation.getCreateDateTime().toEpochMilli()));
        operation.getAccountsSpaceId().ifPresent(id -> builder.setAccountsSpaceId(OperationProviderAccountsSpace
                .newBuilder().setId(id).build()));
        if (operation.getFullErrorMessage().isPresent()) {
            String flattenError = Errors.flattenErrors(operation.getFullErrorMessage().get()
                    .getErrorCollection(locale).toErrorCollection());
            builder.setFailure(OperationErrorDetails.newBuilder().addErrors(flattenError).build());
        } else {
            operation.getErrorMessage().ifPresent(e -> builder.setFailure(OperationErrorDetails.newBuilder()
                    .addErrors(e).build()));
        }
        return builder.build();
    }

    private OperationStatus toStatus(AccountsQuotasOperationsModel operation) {
        if (operation.getRequestStatus().isEmpty()) {
            return OperationStatus.IN_PROGRESS;
        }
        switch (operation.getRequestStatus().get()) {
            case WAITING:
                return OperationStatus.IN_PROGRESS;
            case OK:
                return OperationStatus.SUCCESS;
            case ERROR:
                return OperationStatus.FAILURE;
            default:
                throw new IllegalArgumentException("Unexpected operation status "
                        + operation.getRequestStatus().get());
        }
    }

}
