package ru.yandex.intranet.d.grpc.services;

import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.Collectors;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.backend.service.proto.AccountsSpaceOfProviderReserveAccount;
import ru.yandex.intranet.d.backend.service.proto.GetProviderExternalAccountUrlTemplateRequest;
import ru.yandex.intranet.d.backend.service.proto.GetProviderExternalAccountUrlTemplateResponse;
import ru.yandex.intranet.d.backend.service.proto.GetProviderRelatedResourcesSettingsRequest;
import ru.yandex.intranet.d.backend.service.proto.GetProviderRelatedResourcesSettingsResponse;
import ru.yandex.intranet.d.backend.service.proto.GetProviderRequest;
import ru.yandex.intranet.d.backend.service.proto.GetProviderReserveAccountsRequest;
import ru.yandex.intranet.d.backend.service.proto.GetProviderReserveAccountsResponse;
import ru.yandex.intranet.d.backend.service.proto.ListProvidersRequest;
import ru.yandex.intranet.d.backend.service.proto.ListProvidersResponse;
import ru.yandex.intranet.d.backend.service.proto.ProvideReserveAccountProvisionAmount;
import ru.yandex.intranet.d.backend.service.proto.ProvideReserveAccountRequest;
import ru.yandex.intranet.d.backend.service.proto.ProvideReserveAccountResponse;
import ru.yandex.intranet.d.backend.service.proto.ProvideReserveAccountResult;
import ru.yandex.intranet.d.backend.service.proto.ProvideReserveAccountResultProvision;
import ru.yandex.intranet.d.backend.service.proto.Provider;
import ru.yandex.intranet.d.backend.service.proto.ProviderReserveAccount;
import ru.yandex.intranet.d.backend.service.proto.ProviderUISettings;
import ru.yandex.intranet.d.backend.service.proto.ProvidersPageToken;
import ru.yandex.intranet.d.backend.service.proto.ProvidersServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.RelatedResource;
import ru.yandex.intranet.d.backend.service.proto.RelatedResourcesForResource;
import ru.yandex.intranet.d.backend.service.proto.RelatedResourcesSettings;
import ru.yandex.intranet.d.backend.service.proto.ReserveProvisionOperationStatus;
import ru.yandex.intranet.d.backend.service.proto.SetProviderExternalAccountUrlTemplateRequest;
import ru.yandex.intranet.d.backend.service.proto.SetProviderExternalAccountUrlTemplateResponse;
import ru.yandex.intranet.d.backend.service.proto.SetProviderReadOnlyRequest;
import ru.yandex.intranet.d.backend.service.proto.SetProviderRelatedResourcesSettingsRequest;
import ru.yandex.intranet.d.backend.service.proto.SetProviderRelatedResourcesSettingsResponse;
import ru.yandex.intranet.d.backend.service.proto.SetProviderUISettingsRequest;
import ru.yandex.intranet.d.backend.service.proto.SetProviderUISettingsResponse;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.grpc.GrpcIdempotency;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.services.accounts.ReserveAccountsService;
import ru.yandex.intranet.d.services.providers.ProviderRelatedResourcesService;
import ru.yandex.intranet.d.services.providers.ProvidersService;
import ru.yandex.intranet.d.services.provisions.ReserveProvisionsService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.providers.ExternalAccountUrlTemplateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderReserveAccountsDto;
import ru.yandex.intranet.d.web.model.providers.PutProviderRelatedResourcesSettingsDto;
import ru.yandex.intranet.d.web.model.providers.PutProviderRelatedResourcesSettingsRequestDto;
import ru.yandex.intranet.d.web.model.providers.PutRelatedResourceDto;
import ru.yandex.intranet.d.web.model.providers.PutRelatedResourcesForResourceDto;
import ru.yandex.intranet.d.web.model.provisions.ProviderReserveProvisionRequestValueDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsRequestDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsResponseDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsStatusDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC providers service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcProvidersServiceImpl extends ProvidersServiceGrpc.ProvidersServiceImplBase {

    private final ProvidersService providersService;
    private final ProviderRelatedResourcesService providerRelatedResourcesService;
    private final ReserveAccountsService reserveAccountsService;
    private final ReserveProvisionsService reserveProvisionsService;
    private final MessageSource messages;
    private final CommonProtoConverter converter;

    public GrpcProvidersServiceImpl(ProvidersService providersService,
                                    ProviderRelatedResourcesService providerRelatedResourcesService,
                                    ReserveAccountsService reserveAccountsService,
                                    ReserveProvisionsService reserveProvisionsService,
                                    @Qualifier("messageSource") MessageSource messages,
                                    CommonProtoConverter converter
    ) {
        this.providersService = providersService;
        this.providerRelatedResourcesService = providerRelatedResourcesService;
        this.reserveAccountsService = reserveAccountsService;
        this.reserveProvisionsService = reserveProvisionsService;
        this.messages = messages;
        this.converter = converter;
    }

    @Override
    public void listProviders(ListProvidersRequest request,
                              StreamObserver<ListProvidersResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> providersService.getPage(toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPage(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getProvider(GetProviderRequest request,
                            StreamObserver<Provider> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> providersService.getById(reqParam.getProviderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toProvider(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void setProviderReadOnly(SetProviderReadOnlyRequest request,
                                    StreamObserver<Provider> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> providersService.setReadOnly(reqParam.getProviderId(),
                        reqParam.getReadOnly(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toProvider(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getProviderRelatedResourcesSettings(
            GetProviderRelatedResourcesSettingsRequest request,
            StreamObserver<GetProviderRelatedResourcesSettingsResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> providersService.getById(reqParam.getProviderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toRelatedResourceSettings(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void setProviderRelatedResourcesSettings(
            SetProviderRelatedResourcesSettingsRequest request,
            StreamObserver<SetProviderRelatedResourcesSettingsResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> providerRelatedResourcesService.put(reqParam.getProviderId(),
                                toPutRelatedResources(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toRelatedResourceSettingsSet(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void setProviderUISettings(
            SetProviderUISettingsRequest request,
            StreamObserver<SetProviderUISettingsResponse> responseObserver
    ) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req.flatMap(reqParam ->
                providersService.mergeUiSettings(
                        reqParam.getProviderId(), reqParam.getProviderVersion(),
                        toModel(reqParam.getUiSettings()),
                        currentUser, locale
                ).flatMap(resp -> resp.match(u -> Mono.just(toSetProviderUISettingsResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void setProviderExternalAccountUrlTemplate(
            SetProviderExternalAccountUrlTemplateRequest request,
            StreamObserver<SetProviderExternalAccountUrlTemplateResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req.flatMap(reqParam ->
                providersService.mergeExternalAccountUrlTemplate(
                        reqParam.getProviderId(), reqParam.getProviderVersion(),
                        reqParam.getExternalAccountUrlTemplateList().stream()
                                .map(ExternalAccountUrlTemplateDto::new).toList(),
                        currentUser, locale
                ).flatMap(resp -> resp.match(u -> Mono.just(toSetProviderExternalAccountUrlTemplateResponse(u)),
                        e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getProviderExternalAccountUrlTemplate(
            GetProviderExternalAccountUrlTemplateRequest request,
            StreamObserver<GetProviderExternalAccountUrlTemplateResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req.flatMap(reqParam ->
                providersService.getById(reqParam.getProviderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toGetProviderExternalAccountUrlTemplateResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getProviderReserveAccounts(GetProviderReserveAccountsRequest request,
                                           StreamObserver<GetProviderReserveAccountsResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> reserveAccountsService.findReserveAccountsApiMono(reqParam.getProviderId(),
                                currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toReserveAccounts(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void provideReserveAccount(ProvideReserveAccountRequest request,
                                      StreamObserver<ProvideReserveAccountResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Optional<String> idempotencyKey = GrpcIdempotency.idempotencyKey();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> reserveProvisionsService.provideReserveMono(reqParam.getProviderId(),
                                toProvideRequest(request), idempotencyKey.orElse(null), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toProvideResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private UpdateProviderReserveProvisionsRequestDto toProvideRequest(ProvideReserveAccountRequest request) {
        String accountsSpaceId = request.hasAccountsSpaceId() ? request.getAccountsSpaceId().getId() : null;
        List<ProviderReserveProvisionRequestValueDto> values = request.getValuesList().stream()
                .map(v -> new ProviderReserveProvisionRequestValueDto(v.getResourceId(),
                        v.hasProvidedAmount() ? v.getProvidedAmount().getValue() : null,
                        v.hasProvidedAmount() ? v.getProvidedAmount().getUnitKey() : null))
                .collect(Collectors.toList());
        return new UpdateProviderReserveProvisionsRequestDto(accountsSpaceId, values, request.getDeltaValues());
    }

    private ProvideReserveAccountResponse toProvideResponse(UpdateProviderReserveProvisionsResponseDto response) {
        ProvideReserveAccountResponse.Builder builder = ProvideReserveAccountResponse.newBuilder();
        builder.setOperationId(response.getOperationId());
        builder.setOperationStatus(toOperationStatus(response.getOperationStatus()));
        if (response.getResult() != null) {
            ProvideReserveAccountResult.Builder resultBuilder = ProvideReserveAccountResult.newBuilder();
            resultBuilder.setAccountId(response.getResult().getAccountId());
            resultBuilder.addAllValues(response.getResult().getValues().stream().map(v -> {
                ProvideReserveAccountResultProvision.Builder provisionBuilder = ProvideReserveAccountResultProvision
                        .newBuilder();
                provisionBuilder.setResourceId(v.getResourceId());
                provisionBuilder.setProvidedAmount(ProvideReserveAccountProvisionAmount.newBuilder()
                        .setValue(v.getProvided())
                        .setUnitKey(v.getProvidedUnitKey())
                        .build());
                return provisionBuilder.build();
            }).collect(Collectors.toList()));
            builder.setResult(resultBuilder.build());
        }
        return builder.build();
    }

    private ReserveProvisionOperationStatus toOperationStatus(UpdateProviderReserveProvisionsStatusDto status) {
        return switch (status) {
            case IN_PROGRESS -> ReserveProvisionOperationStatus.RESERVE_PROVISION_IN_PROGRESS;
            case SUCCESS -> ReserveProvisionOperationStatus.RESERVE_PROVISION_SUCCESS;
        };
    }

    private GetProviderReserveAccountsResponse toReserveAccounts(ProviderReserveAccountsDto dto) {
        GetProviderReserveAccountsResponse.Builder builder = GetProviderReserveAccountsResponse.newBuilder();
        dto.getReserveAccounts().forEach(account -> {
            ProviderReserveAccount.Builder accountBuilder = ProviderReserveAccount.newBuilder();
            accountBuilder.setAccountId(account.getAccountId());
            if (account.getAccountsSpaceId() != null) {
                accountBuilder.setAccountsSpaceId(AccountsSpaceOfProviderReserveAccount.newBuilder()
                        .setId(account.getAccountsSpaceId())
                        .build());
            }
            builder.addReserveAccounts(accountBuilder.build());
        });
        return builder.build();
    }

    private ru.yandex.intranet.d.model.providers.ProviderUISettings toModel(ProviderUISettings uiSettings) {
        return new ru.yandex.intranet.d.model.providers.ProviderUISettings(
                converter.toModel(uiSettings.getTitleForTheAccount())
        );
    }

    private SetProviderUISettingsResponse toSetProviderUISettingsResponse(ProviderModel model) {
        SetProviderUISettingsResponse.Builder builder = SetProviderUISettingsResponse.newBuilder();
        builder.setProviderVersion(model.getVersion());
        if (model.getUiSettings().isPresent()) {
            builder.setUiSettings(converter.toProto(model.getUiSettings().get()));
        }
        return builder.build();
    }

    private SetProviderExternalAccountUrlTemplateResponse toSetProviderExternalAccountUrlTemplateResponse(
            ProviderModel model) {
        SetProviderExternalAccountUrlTemplateResponse.Builder builder =
                SetProviderExternalAccountUrlTemplateResponse.newBuilder();
        builder.setProviderVersion(model.getVersion());
        model.getAccountsSettings().getExternalAccountUrlTemplates()
                .forEach(template -> builder.addExternalAccountUrlTemplate(converter.toProto(template)));
        return builder.build();
    }

    private GetProviderExternalAccountUrlTemplateResponse toGetProviderExternalAccountUrlTemplateResponse(
            ProviderModel model) {
        GetProviderExternalAccountUrlTemplateResponse.Builder builder =
                GetProviderExternalAccountUrlTemplateResponse.newBuilder();
        builder.setProviderVersion(model.getVersion());
        model.getAccountsSettings().getExternalAccountUrlTemplates()
                .forEach(template -> builder.addExternalAccountUrlTemplate(converter.toProto(template)));
        return builder.build();
    }

    private PageRequest toPageRequest(ListProvidersRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private ListProvidersResponse toPage(Page<ProviderModel> page, Locale locale) {
        ListProvidersResponse.Builder builder = ListProvidersResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ProvidersPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addProviders(toProvider(e, locale)));
        return builder.build();
    }

    private Provider toProvider(ProviderModel provider, Locale locale) {
        Provider.Builder builder = Provider.newBuilder();
        builder.setProviderId(provider.getId());
        builder.setVersion(provider.getVersion());
        builder.setName(Locales.select(provider.getNameEn(), provider.getNameRu(), locale));
        builder.setDescription(Locales.select(provider.getDescriptionEn(),
                provider.getDescriptionRu(), locale));
        builder.setAbcServiceId(provider.getServiceId());
        builder.setReadOnly(provider.isReadOnly());
        builder.setManaged(provider.isManaged());
        builder.setKey(provider.getKey());
        provider.getUiSettings().ifPresent(it -> builder.setUiSettings(converter.toProto(it)));
        return builder.build();
    }

    private GetProviderRelatedResourcesSettingsResponse toRelatedResourceSettings(ProviderModel provider) {
        GetProviderRelatedResourcesSettingsResponse.Builder builder
                = GetProviderRelatedResourcesSettingsResponse.newBuilder();
        builder.setProviderVersion(provider.getVersion());
        builder.setSettings(toSettings(provider));
        return builder.build();
    }

    private SetProviderRelatedResourcesSettingsResponse toRelatedResourceSettingsSet(ProviderModel provider) {
        SetProviderRelatedResourcesSettingsResponse.Builder builder
                = SetProviderRelatedResourcesSettingsResponse.newBuilder();
        builder.setProviderVersion(provider.getVersion());
        builder.setSettings(toSettings(provider));
        return builder.build();
    }

    private RelatedResourcesSettings toSettings(ProviderModel provider) {
        RelatedResourcesSettings.Builder settingsBuilder = RelatedResourcesSettings.newBuilder();
        provider.getRelatedResourcesByResourceId().ifPresent(map -> map.forEach((k, v) -> {
            RelatedResourcesForResource.Builder resourcesBuilder = RelatedResourcesForResource.newBuilder();
            resourcesBuilder.setResourceId(k);
            v.getRelatedCoefficientMap().forEach((ik, iv) -> {
                RelatedResource.Builder resourceBuilder = RelatedResource.newBuilder();
                resourceBuilder.setResourceId(ik);
                resourceBuilder.setNumerator(iv.getNumerator());
                resourceBuilder.setDenominator(iv.getDenominator());
                resourcesBuilder.addRelatedResources(resourceBuilder.build());
            });
            settingsBuilder.addRelatedResourcesByResource(resourcesBuilder.build());
        }));
        return settingsBuilder.build();
    }

    private PutProviderRelatedResourcesSettingsRequestDto toPutRelatedResources(
            SetProviderRelatedResourcesSettingsRequest request) {
        return new PutProviderRelatedResourcesSettingsRequestDto(new PutProviderRelatedResourcesSettingsDto(
                request.getSettings().getRelatedResourcesByResourceList().stream()
                        .map(v -> new PutRelatedResourcesForResourceDto(v.getResourceId(),
                                v.getRelatedResourcesList().stream()
                                        .map(iv -> new PutRelatedResourceDto(iv.getResourceId(),
                                                iv.getNumerator(), iv.getDenominator()))
                                        .collect(Collectors.toList())))
                        .collect(Collectors.toList())),
                request.getProviderVersion());
    }

}
