package ru.yandex.intranet.d.grpc.services;

import java.math.BigDecimal;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.backend.service.proto.GetResourceProvisionByFolderAccountRequest;
import ru.yandex.intranet.d.backend.service.proto.ListProvisionsByFolderAccountRequest;
import ru.yandex.intranet.d.backend.service.proto.ListProvisionsByFolderAccountResponse;
import ru.yandex.intranet.d.backend.service.proto.ProviderProvision;
import ru.yandex.intranet.d.backend.service.proto.ProvisionAmount;
import ru.yandex.intranet.d.backend.service.proto.ProvisionOperationStatus;
import ru.yandex.intranet.d.backend.service.proto.ProvisionsPageToken;
import ru.yandex.intranet.d.backend.service.proto.ProvisionsServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.UpdateProvisionsRequest;
import ru.yandex.intranet.d.backend.service.proto.UpdateProvisionsResponse;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.grpc.GrpcIdempotency;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.provisions.ProvisionsReadService;
import ru.yandex.intranet.d.services.quotas.ProvisionLogicService;
import ru.yandex.intranet.d.services.quotas.ProvisionOperationResult;
import ru.yandex.intranet.d.services.quotas.ProvisionOperationResultStatus;
import ru.yandex.intranet.d.services.quotas.ProvisionService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.quotas.ProvisionLiteDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsRequestDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC provisions service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcProvisionsServiceImpl extends ProvisionsServiceGrpc.ProvisionsServiceImplBase {

    private final ProvisionsReadService provisionsReadService;
    private final ProvisionService provisionService;
    private final ProvisionLogicService provisionLogicService;
    private final MessageSource messages;

    public GrpcProvisionsServiceImpl(ProvisionsReadService provisionsReadService,
                                     ProvisionService provisionService,
                                     ProvisionLogicService provisionLogicService,
                                     @Qualifier("messageSource") MessageSource messages) {
        this.provisionsReadService = provisionsReadService;
        this.provisionService = provisionService;
        this.provisionLogicService = provisionLogicService;
        this.messages = messages;
    }

    @Override
    public void listProvisionsByFolderAccount(ListProvisionsByFolderAccountRequest request,
                                              StreamObserver<ListProvisionsByFolderAccountResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> provisionsReadService.getAccountProvisions(reqParam.getFolderId(),
                                reqParam.getAccountId(), toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPageByFolder(u.getQuotas(), u.getResources(),
                                        u.getUnitsEnsembles())),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getResourceProvisionByFolderAccount(GetResourceProvisionByFolderAccountRequest request,
                                                    StreamObserver<ProviderProvision> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> provisionsReadService.getOneProvision(reqParam.getFolderId(),
                                reqParam.getAccountId(), reqParam.getProviderId(), reqParam.getResourceId(),
                                currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toProvision(u.getQuotas(), u.getResources(),
                                        u.getUnitsEnsembles())),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void updateProvisions(UpdateProvisionsRequest request,
                                 StreamObserver<UpdateProvisionsResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Optional<String> idempotencyKey = GrpcIdempotency.idempotencyKey();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> provisionLogicService.updateProvisionMono(toUpdateRequest(request), currentUser,
                                locale, true, idempotencyKey.orElse(null))
                        .flatMap(resp -> resp.match(u -> Mono.just(toUpdateResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private UpdateProvisionsRequestDto toUpdateRequest(UpdateProvisionsRequest request) {
        UpdateProvisionsRequestDto.Builder result = UpdateProvisionsRequestDto.builder();
        result.setAccountId(request.getAccountId());
        result.setFolderId(request.getFolderId());
        result.setUpdatedProvisions(request.getUpdatedProvisionsList().stream().map(p -> {
            ProvisionLiteDto.Builder builder = ProvisionLiteDto.builder();
            builder.resourceId(p.getResourceId());
            // Not nice but... provider id is unused, it is not strictly necessary anyway...
            if (p.hasProvidedAmount()) {
                builder.providedAmount(String.valueOf(p.getProvidedAmount().getValue()));
                // Not nice but... implementation should interpret this differently for public API and front API
                builder.providedAmountUnitId(p.getProvidedAmount().getUnitKey());
                // Old values are not supplied through public API, that's by design
            }
            return builder.build();
        }).collect(Collectors.toList()));
        return result.build();
    }

    private UpdateProvisionsResponse toUpdateResponse(ProvisionOperationResult response) {
        UpdateProvisionsResponse.Builder result = UpdateProvisionsResponse.newBuilder();
        result.setOperationId(response.getOperationId());
        result.setOperationStatus(toStatus(response.getStatus()));
        response.getExpandedProvisionResult().ifPresent(expanded -> {
            AccountModel account = expanded.getCurrentAccount();
            String accountId = account.getId();
            String folderId = account.getFolderId();
            String providerId = account.getProviderId();
            Set<String> resourceIdsToReturn = new HashSet<>();
            Map<String, AccountsQuotasModel> updatedQuotasById = expanded.getActualUpdatedQuotas().stream()
                    .filter(q -> q.getAccountId().equals(accountId)).collect(Collectors
                            .toMap(AccountsQuotasModel::getResourceId, Function.identity()));
            Map<String, AccountsQuotasModel> originalQuotasById = expanded.getCurrentActualQuotas().stream()
                    .filter(q -> q.getAccountId().equals(accountId)).collect(Collectors
                            .toMap(AccountsQuotasModel::getResourceId, Function.identity()));
            Set<String> updatedAndOriginalResources = Sets.union(updatedQuotasById.keySet(),
                    originalQuotasById.keySet());
            updatedAndOriginalResources.forEach(resourceId -> {
                Optional<Long> updatedProvided = Optional.ofNullable(updatedQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                Optional<Long> originalProvided = Optional.ofNullable(originalQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                boolean noUpdateAndNonZero = updatedProvided.isEmpty()
                        && originalProvided.map(v -> v != 0).orElse(false);
                if (noUpdateAndNonZero || updatedProvided.isPresent()) {
                    resourceIdsToReturn.add(resourceId);
                }
            });
            resourceIdsToReturn.addAll(expanded.getRequestResourceModels().stream().map(ResourceModel::getId)
                    .collect(Collectors.toList()));
            resourceIdsToReturn.forEach(resourceId -> {
                ResourceModel resource = expanded.getAllResourceByIdMap().get(resourceId);
                UnitsEnsembleModel unitsEnsemble = expanded.getEnsembleModelByIdMap()
                        .get(resource.getUnitsEnsembleId());
                Optional<Long> updatedProvided = Optional.ofNullable(updatedQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                Optional<Long> originalProvided = Optional.ofNullable(originalQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                long providedResult = updatedProvided.or(() -> originalProvided).orElse(0L);
                long allocatedResult = updatedProvided.isPresent()
                        ? Optional.ofNullable(updatedQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getAllocatedQuota()).orElse(0L)).orElse(0L)
                        : Optional.ofNullable(originalQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getAllocatedQuota()).orElse(0L)).orElse(0L);
                Tuple2<BigDecimal, UnitModel> convertedProvided = Units
                        .convertToApi(providedResult, resource, unitsEnsemble);
                Tuple2<BigDecimal, UnitModel> convertedAllocated = Units
                        .convertToApi(allocatedResult, resource, unitsEnsemble);
                ProviderProvision.Builder builder = ProviderProvision.newBuilder();
                builder.setFolderId(folderId);
                builder.setAccountId(accountId);
                builder.setProviderId(providerId);
                builder.setResourceId(resourceId);
                builder.setProvidedAmount(ProvisionAmount.newBuilder()
                        .setValue(convertedProvided.getT1().longValueExact())
                        .setUnitKey(convertedProvided.getT2().getKey())
                        .build());
                builder.setAllocatedAmount(ProvisionAmount.newBuilder()
                        .setValue(convertedAllocated.getT1().longValueExact())
                        .setUnitKey(convertedAllocated.getT2().getKey())
                        .build());
                result.addProvisions(builder.build());
            });

        });
        return result.build();
    }

    private ProvisionOperationStatus toStatus(ProvisionOperationResultStatus status) {
        switch (status) {
            case SUCCESS:
                return ProvisionOperationStatus.PROVISION_SUCCESS;
            case IN_PROGRESS:
                return ProvisionOperationStatus.PROVISION_IN_PROGRESS;
            default:
                throw new IllegalArgumentException("Unexpected operation status: " + status);
        }
    }

    private ProviderProvision toProvision(AccountsQuotasModel provision, Map<String, ResourceModel> resources,
                                          Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ResourceModel resource = resources.get(provision.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        long provided = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
        long allocated = provision.getAllocatedQuota() != null ? provision.getAllocatedQuota() : 0L;
        Tuple2<BigDecimal, UnitModel> convertedProvided = Units.convertToApi(provided, resource, unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> convertedAllocated = Units.convertToApi(allocated, resource, unitsEnsemble);
        ProviderProvision.Builder builder = ProviderProvision.newBuilder();
        builder.setFolderId(provision.getFolderId());
        builder.setAccountId(provision.getAccountId());
        builder.setProviderId(provision.getProviderId());
        builder.setResourceId(provision.getResourceId());
        builder.setProvidedAmount(ProvisionAmount.newBuilder().setValue(convertedProvided.getT1().longValue())
                .setUnitKey(convertedProvided.getT2().getKey()).build());
        builder.setAllocatedAmount(ProvisionAmount.newBuilder().setValue(convertedAllocated.getT1().longValue())
                .setUnitKey(convertedAllocated.getT2().getKey()).build());
        return builder.build();
    }

    private PageRequest toPageRequest(ListProvisionsByFolderAccountRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private ListProvisionsByFolderAccountResponse toPageByFolder(Page<AccountsQuotasModel> page,
                                                                 Map<String, ResourceModel> resources,
                                                                 Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ListProvisionsByFolderAccountResponse.Builder builder = ListProvisionsByFolderAccountResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ProvisionsPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addProvisions(toProvision(e, resources, unitsEnsembles)));
        return builder.build();
    }

}
