package ru.yandex.intranet.d.grpc.services;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.AccountsSpaceIdentity;
import ru.yandex.intranet.d.backend.service.proto.FailedImport;
import ru.yandex.intranet.d.backend.service.proto.ImportFieldError;
import ru.yandex.intranet.d.backend.service.proto.ImportQuotasRequest;
import ru.yandex.intranet.d.backend.service.proto.ImportQuotasResponse;
import ru.yandex.intranet.d.backend.service.proto.QuotasImportsServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.SuccessfulImport;
import ru.yandex.intranet.d.backend.service.proto.TargetFolder;
import ru.yandex.intranet.d.backend.service.proto.TargetService;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.services.imports.QuotasImportsService;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.imports.AccountSpaceIdentityDto;
import ru.yandex.intranet.d.web.model.imports.ImportAccountDto;
import ru.yandex.intranet.d.web.model.imports.ImportAccountProvisionDto;
import ru.yandex.intranet.d.web.model.imports.ImportDto;
import ru.yandex.intranet.d.web.model.imports.ImportFolderDto;
import ru.yandex.intranet.d.web.model.imports.ImportResourceDto;
import ru.yandex.intranet.d.web.model.imports.ImportResultDto;
import ru.yandex.intranet.d.web.model.imports.ResourceIdentityDto;
import ru.yandex.intranet.d.web.model.imports.SegmentKey;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC quotas imports service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcQuotasImportsServiceImpl extends QuotasImportsServiceGrpc.QuotasImportsServiceImplBase {

    private final QuotasImportsService quotasImportsService;
    private final MessageSource messages;

    public GrpcQuotasImportsServiceImpl(QuotasImportsService quotasImportsService,
                                        @Qualifier("messageSource") MessageSource messages) {
        this.quotasImportsService = quotasImportsService;
        this.messages = messages;
    }

    @Override
    public void importQuotas(ImportQuotasRequest request, StreamObserver<ImportQuotasResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> quotasImportsService.importQuotas(toImportRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toImportResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private ImportDto toImportRequest(ImportQuotasRequest request) {
        List<ImportFolderDto> foldersQuotas = new ArrayList<>();
        request.getQuotasList().forEach(folderQuotas -> {
            String folderId = folderQuotas.hasTargetFolder() ? folderQuotas.getTargetFolder().getFolderId() : null;
            Long serviceId = folderQuotas.hasTargetService() ? folderQuotas.getTargetService().getServiceId() : null;
            List<ImportResourceDto> resourceQuotas = new ArrayList<>();
            List<ImportAccountDto> accounts = new ArrayList<>();
            folderQuotas.getResourceQuotasList().forEach(resourceQuota -> {
                String resourceId = null;
                ResourceIdentityDto resourceIdentity = null;
                switch (resourceQuota.getIdentityCase()) {
                    case RESOURCE_ID:
                        resourceId = resourceQuota.getResourceId();
                        break;
                    case RESOURCE_IDENTITY:
                        resourceIdentity = toResourceIdentityDto(resourceQuota.getResourceIdentity());
                        break;
                    case IDENTITY_NOT_SET:
                    default:
                        break;
                }
                resourceQuotas.add(new ImportResourceDto(resourceId,
                        resourceIdentity, resourceQuota.getProviderId(),
                        resourceQuota.hasQuota() ? resourceQuota.getQuota().getValue() : null,
                        resourceQuota.hasQuota() ? resourceQuota.getQuota().getUnitKey() : null,
                        resourceQuota.hasBalance() ? resourceQuota.getBalance().getValue() : null,
                        resourceQuota.hasBalance() ? resourceQuota.getBalance().getUnitKey() : null));
            });
            folderQuotas.getAccountsList().forEach(account -> {
                List<ImportAccountProvisionDto> provisions = new ArrayList<>();
                account.getProvisionsList().forEach(provision -> {
                    String resourceId = null;
                    ResourceIdentityDto resourceIdentity = null;
                    switch (provision.getIdentityCase()) {
                        case RESOURCE_ID:
                            resourceId = provision.getResourceId();
                            break;
                        case RESOURCE_IDENTITY:
                            resourceIdentity = toResourceIdentityDto(provision.getResourceIdentity());
                            break;
                        case IDENTITY_NOT_SET:
                        default:
                            break;
                    }
                    provisions.add(new ImportAccountProvisionDto(resourceId, resourceIdentity,
                            provision.hasProvided() ? provision.getProvided().getValue() : null,
                            provision.hasProvided() ? provision.getProvided().getUnitKey() : null,
                            provision.hasAllocated() ? provision.getAllocated().getValue() : null,
                            provision.hasAllocated() ? provision.getAllocated().getUnitKey() : null));
                });
                String displayName = !account.getDisplayName().isEmpty() ? account.getDisplayName() : null;
                accounts.add(new ImportAccountDto(account.getAccountId(), account.getKey(),
                        displayName, account.getDeleted(), account.getProviderId(), provisions,
                        toDto(account.getAccountsSpace()), account.getFreeTier()));
            });
            foldersQuotas.add(new ImportFolderDto(folderId, serviceId, resourceQuotas, accounts));
        });
        return new ImportDto(foldersQuotas);
    }

    private AccountSpaceIdentityDto toDto(AccountsSpaceIdentity accountSpace) {
        switch (accountSpace.getIdentityCase()) {
            case ACCOUNTS_SPACE_ID:
                return new AccountSpaceIdentityDto(accountSpace.getAccountsSpaceId(), null);
            case SEGMENTS:
                return new AccountSpaceIdentityDto(null,
                        accountSpace.getSegments().getKeyList().stream()
                                .map(s -> new SegmentKey(s.getSegmentationKey(), s.getSegmentKey()))
                                .collect(Collectors.toList())
                );
            case IDENTITY_NOT_SET:
            default:
                return null;
        }
    }

    private ResourceIdentityDto toResourceIdentityDto(
            ru.yandex.intranet.d.backend.service.proto.ResourceIdentity resourceIdentity) {
        return new ResourceIdentityDto(
                resourceIdentity.getResourceTypeKey(),
                toDto(resourceIdentity.getAccountsSpace()),
                resourceIdentity.getResourceSegmentsList().stream()
                        .map(key -> new SegmentKey(key.getSegmentationKey(), key.getSegmentKey()))
                        .collect(Collectors.toList()));
    }

    private ImportQuotasResponse toImportResponse(ImportResultDto response) {
        ImportQuotasResponse.Builder builder = ImportQuotasResponse.newBuilder();
        response.getSuccessfullyImported().forEach(success -> {
            SuccessfulImport.Builder successBuilder = SuccessfulImport.newBuilder();
            if (success.getFolderId().isPresent()) {
                successBuilder.setTargetFolder(TargetFolder.newBuilder()
                        .setFolderId(success.getFolderId().get()).build());
            }
            if (success.getServiceId().isPresent()) {
                successBuilder.setTargetService(TargetService.newBuilder()
                        .setServiceId(success.getServiceId().get()).build());
            }
            builder.addSuccessfullyImported(successBuilder.build());
        });
        response.getImportFailures().forEach(failure -> {
            FailedImport.Builder failureBuilder = FailedImport.newBuilder();
            if (failure.getFolderId().isPresent()) {
                failureBuilder.setTargetFolder(TargetFolder.newBuilder()
                        .setFolderId(failure.getFolderId().get()).build());
            }
            if (failure.getServiceId().isPresent()) {
                failureBuilder.setTargetService(TargetService.newBuilder()
                        .setServiceId(failure.getServiceId().get()).build());
            }
            failure.getErrors().forEach(failureBuilder::addErrors);
            failure.getFieldErrors().forEach((fieldKey, fieldErrors) -> {
                ImportFieldError.Builder fieldErrorBuilder = ImportFieldError.newBuilder();
                fieldErrorBuilder.setKey(fieldKey);
                fieldErrors.forEach(fieldErrorBuilder::addErrors);
                failureBuilder.addFieldErrors(fieldErrorBuilder.build());
            });
            builder.addImportFailures(failureBuilder.build());
        });
        return builder.build();
    }

}
