package ru.yandex.intranet.d.grpc.services;

import java.math.BigDecimal;
import java.util.Locale;
import java.util.Map;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.backend.service.proto.Amount;
import ru.yandex.intranet.d.backend.service.proto.FolderQuota;
import ru.yandex.intranet.d.backend.service.proto.FolderResourceQuota;
import ru.yandex.intranet.d.backend.service.proto.GetQuotaByFolderResourceRequest;
import ru.yandex.intranet.d.backend.service.proto.ListQuotasByFolderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListQuotasByFolderResponse;
import ru.yandex.intranet.d.backend.service.proto.QuotasPageToken;
import ru.yandex.intranet.d.backend.service.proto.QuotasServiceGrpc;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.quotas.ExpandedQuotas;
import ru.yandex.intranet.d.services.quotas.QuotasService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC quotas service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcQuotasServiceImpl extends QuotasServiceGrpc.QuotasServiceImplBase {

    private final QuotasService quotasService;
    private final MessageSource messages;

    public GrpcQuotasServiceImpl(QuotasService quotasService, @Qualifier("messageSource") MessageSource messages) {
        this.quotasService = quotasService;
        this.messages = messages;
    }

    @Override
    public void listQuotasByFolder(ListQuotasByFolderRequest request,
                                   StreamObserver<ListQuotasByFolderResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> quotasService.getPageForFolder(reqParam.getFolderId(),
                        toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPage(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getQuotaByFolderResource(GetQuotaByFolderResourceRequest request,
                                         StreamObserver<FolderResourceQuota> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> quotasService.getQuotaForFolderAndResource(reqParam.getFolderId(),
                        reqParam.getResourceId(), reqParam.getProviderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toFolderResourceQuota(u.getQuotas(),
                                u.getResources(), u.getUnitsEnsembles())),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private PageRequest toPageRequest(ListQuotasByFolderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private ListQuotasByFolderResponse toPage(ExpandedQuotas<Page<QuotaModel>> page) {
        ListQuotasByFolderResponse.Builder builder = ListQuotasByFolderResponse.newBuilder();
        page.getQuotas().getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(QuotasPageToken.newBuilder().setToken(t).build()));
        page.getQuotas().getItems().forEach(e -> builder.addQuotas(toFolderQuota(e, page.getResources(),
                page.getUnitsEnsembles())));
        return builder.build();
    }

    private FolderQuota toFolderQuota(QuotaModel quota, Map<String, ResourceModel> resources,
                                      Map<String, UnitsEnsembleModel> unitsEnsembles) {
        FolderQuota.Builder builder = FolderQuota.newBuilder();
        ResourceModel resource = resources.get(quota.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedQuota = Units.convertToApi(quota.getQuota(), resource,
                unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> convertedBalance = Units.convertToApi(quota.getBalance(), resource,
                unitsEnsemble);
        builder.setResourceId(quota.getResourceId());
        builder.setQuota(Amount.newBuilder().setValue(convertedQuota.getT1().longValue())
                .setUnitKey(convertedQuota.getT2().getKey()).build());
        builder.setBalance(Amount.newBuilder().setValue(convertedBalance.getT1().longValue())
                .setUnitKey(convertedBalance.getT2().getKey()).build());
        builder.setProviderId(quota.getProviderId());
        return builder.build();
    }

    private FolderResourceQuota toFolderResourceQuota(QuotaModel quota, Map<String, ResourceModel> resources,
                                                      Map<String, UnitsEnsembleModel> unitsEnsembles) {
        FolderResourceQuota.Builder builder = FolderResourceQuota.newBuilder();
        ResourceModel resource = resources.get(quota.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedQuota = Units.convertToApi(quota.getQuota(), resource,
                unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> convertedBalance = Units.convertToApi(quota.getBalance(), resource,
                unitsEnsemble);
        builder.setQuota(Amount.newBuilder().setValue(convertedQuota.getT1().longValue())
                .setUnitKey(convertedQuota.getT2().getKey()).build());
        builder.setBalance(Amount.newBuilder().setValue(convertedBalance.getT1().longValue())
                .setUnitKey(convertedBalance.getT2().getKey()).build());
        return builder.build();
    }

}
