package ru.yandex.intranet.d.grpc.services;

import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.protobuf.Descriptors;
import com.google.protobuf.Empty;
import com.google.protobuf.FieldMask;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.CreateAccountsSpaceRequest;
import ru.yandex.intranet.d.backend.service.proto.CreateResourceRequest;
import ru.yandex.intranet.d.backend.service.proto.CreateResourceSegmentRequest;
import ru.yandex.intranet.d.backend.service.proto.CreateResourceSegmentationRequest;
import ru.yandex.intranet.d.backend.service.proto.CreateResourceTypeRequest;
import ru.yandex.intranet.d.backend.service.proto.DeleteAccountsSpaceRequest;
import ru.yandex.intranet.d.backend.service.proto.DeleteResourceRequest;
import ru.yandex.intranet.d.backend.service.proto.DeleteResourceSegmentRequest;
import ru.yandex.intranet.d.backend.service.proto.DeleteResourceSegmentationRequest;
import ru.yandex.intranet.d.backend.service.proto.DeleteResourceTypeRequest;
import ru.yandex.intranet.d.backend.service.proto.FullAccountsSpace;
import ru.yandex.intranet.d.backend.service.proto.FullResource;
import ru.yandex.intranet.d.backend.service.proto.FullResourceSegment;
import ru.yandex.intranet.d.backend.service.proto.FullResourceSegmentation;
import ru.yandex.intranet.d.backend.service.proto.FullResourceType;
import ru.yandex.intranet.d.backend.service.proto.GetFullAccountsSpaceRequest;
import ru.yandex.intranet.d.backend.service.proto.GetFullResourceRequest;
import ru.yandex.intranet.d.backend.service.proto.GetResourceSegmentRequest;
import ru.yandex.intranet.d.backend.service.proto.GetResourceSegmentationRequest;
import ru.yandex.intranet.d.backend.service.proto.GetResourceTypeRequest;
import ru.yandex.intranet.d.backend.service.proto.ListFullAccountsSpacesByProviderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListFullAccountsSpacesByProviderResponse;
import ru.yandex.intranet.d.backend.service.proto.ListFullResourcesByProviderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListFullResourcesByProviderResponse;
import ru.yandex.intranet.d.backend.service.proto.ListResourceSegmentationsByProviderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListResourceSegmentationsByProviderResponse;
import ru.yandex.intranet.d.backend.service.proto.ListResourceSegmentsBySegmentationRequest;
import ru.yandex.intranet.d.backend.service.proto.ListResourceSegmentsBySegmentationResponse;
import ru.yandex.intranet.d.backend.service.proto.ListResourceTypesByProviderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListResourceTypesByProviderResponse;
import ru.yandex.intranet.d.backend.service.proto.NewAccountsSpace;
import ru.yandex.intranet.d.backend.service.proto.NewResource;
import ru.yandex.intranet.d.backend.service.proto.NewResourceSegment;
import ru.yandex.intranet.d.backend.service.proto.NewResourceSegmentation;
import ru.yandex.intranet.d.backend.service.proto.NewResourceType;
import ru.yandex.intranet.d.backend.service.proto.PatchResourceTypeRequest;
import ru.yandex.intranet.d.backend.service.proto.ResourceSegmentIds;
import ru.yandex.intranet.d.backend.service.proto.ResourceSegmentationsIds;
import ru.yandex.intranet.d.backend.service.proto.ResourcesManagementPageToken;
import ru.yandex.intranet.d.backend.service.proto.ResourcesManagementServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.SegmentationUISettings;
import ru.yandex.intranet.d.backend.service.proto.SetAccountsSpaceReadOnlyRequest;
import ru.yandex.intranet.d.backend.service.proto.SetResourceReadOnlyRequest;
import ru.yandex.intranet.d.backend.service.proto.UpdateAccountsSpaceRequest;
import ru.yandex.intranet.d.backend.service.proto.UpdateResourceRequest;
import ru.yandex.intranet.d.backend.service.proto.UpdateResourceSegmentRequest;
import ru.yandex.intranet.d.backend.service.proto.UpdateResourceSegmentationRequest;
import ru.yandex.intranet.d.backend.service.proto.UpdateResourceTypeRequest;
import ru.yandex.intranet.d.backend.service.proto.UpdatedAccountsSpace;
import ru.yandex.intranet.d.backend.service.proto.UpdatedResource;
import ru.yandex.intranet.d.backend.service.proto.UpdatedResourceSegment;
import ru.yandex.intranet.d.backend.service.proto.UpdatedResourceSegmentation;
import ru.yandex.intranet.d.backend.service.proto.UpdatedResourceType;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.services.resources.AccountsSpacesService;
import ru.yandex.intranet.d.services.resources.ResourcesService;
import ru.yandex.intranet.d.services.resources.segmentations.ResourceSegmentationsService;
import ru.yandex.intranet.d.services.resources.segments.ResourceSegmentsService;
import ru.yandex.intranet.d.services.resources.types.ResourceTypesService;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.FeatureStateHelper;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.resources.directory.CreateResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.directory.CreateResourceTypeSegmentsDto;
import ru.yandex.intranet.d.web.model.resources.directory.ResourceCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.ResourcePutDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.ResourceSegmentationCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.ResourceSegmentationPutDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.SegmentationUISettingsDto;
import ru.yandex.intranet.d.web.model.resources.directory.segments.ResourceSegmentCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.segments.ResourceSegmentPutDto;
import ru.yandex.intranet.d.web.model.resources.directory.spaces.AccountsSpaceCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.spaces.AccountsSpacePutDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypeCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypePatchDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypePutDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC resources management service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcResourcesManagementServiceImpl
        extends ResourcesManagementServiceGrpc.ResourcesManagementServiceImplBase {

    private final ResourceTypesService resourceTypesService;
    private final ResourceSegmentationsService resourceSegmentationsService;
    private final ResourceSegmentsService resourceSegmentsService;
    private final ResourcesService resourcesService;
    private final MessageSource messages;
    private final AccountsSpacesService accountsSpacesService;
    private final CommonProtoConverter converter;

    public GrpcResourcesManagementServiceImpl(ResourceTypesService resourceTypesService,
                                              ResourceSegmentationsService resourceSegmentationsService,
                                              ResourceSegmentsService resourceSegmentsService,
                                              ResourcesService resourcesService,
                                              @Qualifier("messageSource") MessageSource messages,
                                              AccountsSpacesService accountsSpacesService,
                                              CommonProtoConverter converter
    ) {
        this.resourceTypesService = resourceTypesService;
        this.resourceSegmentationsService = resourceSegmentationsService;
        this.resourceSegmentsService = resourceSegmentsService;
        this.resourcesService = resourcesService;
        this.messages = messages;
        this.accountsSpacesService = accountsSpacesService;
        this.converter = converter;
    }

    @Override
    public void listResourcesByProvider(ListFullResourcesByProviderRequest request,
                                        StreamObserver<ListFullResourcesByProviderResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourcesService.getPage(reqParam.getProviderId(),
                        toPageRequest(reqParam), currentUser, locale, true, false)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPageResource(u.getResources())),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void listResourceTypesByProvider(ListResourceTypesByProviderRequest request,
                                            StreamObserver<ListResourceTypesByProviderResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourceTypesService.getPage(reqParam.getProviderId(),
                        toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPageType(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void listResourceSegmentationsByProvider(
            ListResourceSegmentationsByProviderRequest request,
            StreamObserver<ListResourceSegmentationsByProviderResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourceSegmentationsService.getPage(reqParam.getProviderId(),
                        toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPageSegmentation(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void listResourceSegmentsBySegmentation(
            ListResourceSegmentsBySegmentationRequest request,
            StreamObserver<ListResourceSegmentsBySegmentationResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourceSegmentsService.getPage(reqParam.getProviderId(),
                        reqParam.getResourceSegmentationId(), toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPageSegment(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void listAccountsSpacesByProvider(
            ListFullAccountsSpacesByProviderRequest request,
            StreamObserver<ListFullAccountsSpacesByProviderResponse> responseObserver
    ) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountsSpacesService.getPage(
                        reqParam.getProviderId(), toPageRequest(reqParam), currentUser, locale, false
                ).flatMap(resp -> resp.match(u -> Mono.just(toFullAccountsSpaces(u)),
                        e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getResource(GetFullResourceRequest request, StreamObserver<FullResource> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourcesService.getById(reqParam.getResourceId(),
                        reqParam.getProviderId(), currentUser, locale, true, false)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResource(u.getResources())),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getResourceType(GetResourceTypeRequest request, StreamObserver<FullResourceType> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourceTypesService.getById(reqParam.getResourceTypeId(),
                        reqParam.getProviderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceType(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getResourceSegmentation(GetResourceSegmentationRequest request,
                                        StreamObserver<FullResourceSegmentation> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourceSegmentationsService.getById(reqParam.getResourceSegmentationId(),
                        reqParam.getProviderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceSegmentation(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getResourceSegment(GetResourceSegmentRequest request,
                                   StreamObserver<FullResourceSegment> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourceSegmentsService.getById(reqParam.getResourceSegmentId(),
                        reqParam.getProviderId(), reqParam.getResourceSegmentationId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceSegment(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getAccountsSpace(
            GetFullAccountsSpaceRequest request, StreamObserver<FullAccountsSpace> responseObserver
    ) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountsSpacesService.getById(reqParam.getAccountsSpaceId(),
                        reqParam.getProviderId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toProto(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void setResourceReadOnly(SetResourceReadOnlyRequest request,
                                    StreamObserver<FullResource> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> resourcesService.setReadOnly(reqParam.getResourceId(),
                        reqParam.getProviderId(), reqParam.getReadOnly(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResource(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void deleteResource(DeleteResourceRequest request, StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        // TODO Implement this
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> Mono.error(new StatusRuntimeException(Status.UNIMPLEMENTED))), messages);
    }

    @Override
    public void deleteResourceType(DeleteResourceTypeRequest request, StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        // TODO Implement this
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> Mono.error(new StatusRuntimeException(Status.UNIMPLEMENTED))), messages);
    }

    @Override
    public void deleteResourceSegmentation(DeleteResourceSegmentationRequest request,
                                           StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        // TODO Implement this
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> Mono.error(new StatusRuntimeException(Status.UNIMPLEMENTED))), messages);
    }

    @Override
    public void deleteResourceSegment(DeleteResourceSegmentRequest request, StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        // TODO Implement this
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> Mono.error(new StatusRuntimeException(Status.UNIMPLEMENTED))), messages);
    }

    @Override
    public void deleteAccountsSpace(DeleteAccountsSpaceRequest request, StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        // TODO Implement this
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> Mono.error(new StatusRuntimeException(Status.UNIMPLEMENTED))), messages);
    }

    @Override
    public void setAccountsSpaceReadOnly(
            SetAccountsSpaceReadOnlyRequest request, StreamObserver<FullAccountsSpace> responseObserver
    ) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> accountsSpacesService.setReadOnly(reqParam.getAccountsSpaceId(),
                        reqParam.getProviderId(), reqParam.getReadOnly(), currentUser, locale)
                        .flatMap(resp -> resp.match(m -> Mono.just(toProto(m)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createResource(CreateResourceRequest request, StreamObserver<FullResource> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateCreateRequest(reqParam, locale).andThenMono(rt ->
                        resourcesService.create(toResourceCreate(rt), rt.getProviderId(), currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResource(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createResourceType(CreateResourceTypeRequest request,
                                   StreamObserver<FullResourceType> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateCreateRequest(reqParam, locale).andThenMono(rt ->
                        resourceTypesService.create(toResourceTypeCreate(rt), rt.getProviderId(), currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceType(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createResourceSegmentation(CreateResourceSegmentationRequest request,
                                           StreamObserver<FullResourceSegmentation> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateCreateRequest(reqParam, locale).andThenMono(rt ->
                        resourceSegmentationsService.create(toResourceSegmentationCreate(rt),
                                rt.getProviderId(), currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceSegmentation(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createResourceSegment(CreateResourceSegmentRequest request,
                                      StreamObserver<FullResourceSegment> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateCreateRequest(reqParam, locale).andThenMono(rt ->
                        resourceSegmentsService.create(toResourceSegmentCreate(rt), reqParam.getProviderId(),
                                rt.getResourceSegmentationId(), currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceSegment(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createAccountsSpace(
            CreateAccountsSpaceRequest request, StreamObserver<FullAccountsSpace> responseObserver
    ) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateCreateRequest(reqParam, locale).andThenMono(as ->
                        accountsSpacesService.create(
                                toAccountsSpaceCreateDto(as), as.getProviderId(), currentUser, locale))
                        .flatMap(resp -> resp.match(m -> Mono.just(toProto(m)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void updateResource(UpdateResourceRequest request, StreamObserver<FullResource> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateUpdateRequest(reqParam, locale).andThenMono(rt ->
                        resourcesService.put(reqParam.getResourceId(), reqParam.getProviderId(),
                                reqParam.getVersion(), toResourceUpdate(rt), currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResource(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void updateResourceType(UpdateResourceTypeRequest request,
                                   StreamObserver<FullResourceType> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateUpdateRequest(reqParam, locale).andThenMono(rt ->
                        resourceTypesService.put(reqParam.getResourceTypeId(), reqParam.getProviderId(),
                                reqParam.getVersion(), toResourceTypeUpdate(rt), currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceType(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void patchResourceType(PatchResourceTypeRequest request,
                                  StreamObserver<FullResourceType> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validatePatchRequest(reqParam, locale).andThenMono(rt ->
                        resourceTypesService.patch(reqParam.getResourceTypeId(), reqParam.getProviderId(),
                                reqParam.getVersion(), toResourceTypePatch(rt, reqParam.getFieldMask()),
                                currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceType(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void updateResourceSegmentation(UpdateResourceSegmentationRequest request,
                                           StreamObserver<FullResourceSegmentation> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateUpdateRequest(reqParam, locale).andThenMono(rt ->
                        resourceSegmentationsService.put(reqParam.getResourceSegmentationId(),
                                reqParam.getProviderId(), reqParam.getVersion(), toResourceSegmentationUpdate(rt),
                                currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceSegmentation(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void updateResourceSegment(UpdateResourceSegmentRequest request,
                                      StreamObserver<FullResourceSegment> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateUpdateRequest(reqParam, locale).andThenMono(rt ->
                        resourceSegmentsService.put(reqParam.getResourceSegmentId(), reqParam.getProviderId(),
                                reqParam.getResourceSegmentationId(), reqParam.getVersion(),
                                toResourceSegmentUpdate(rt), currentUser, locale))
                        .flatMap(resp -> resp.match(u -> Mono.just(toResourceSegment(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void updateAccountsSpace(
            UpdateAccountsSpaceRequest request,
            StreamObserver<FullAccountsSpace> responseObserver
    ) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> validateUpdateRequest(reqParam, locale).andThenMono(rt ->
                        accountsSpacesService.put(reqParam.getAccountsSpaceId(),
                                reqParam.getProviderId(), reqParam.getVersion(), toAccountsSpacePutDto(rt),
                                currentUser, locale))
                        .flatMap(resp -> resp.match(m -> Mono.just(toProto(m)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private PageRequest toPageRequest(ListResourceTypesByProviderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private PageRequest toPageRequest(ListResourceSegmentationsByProviderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private PageRequest toPageRequest(ListResourceSegmentsBySegmentationRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private PageRequest toPageRequest(ListFullResourcesByProviderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private PageRequest toPageRequest(ListFullAccountsSpacesByProviderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private ListResourceTypesByProviderResponse toPageType(Page<ResourceTypeModel> page) {
        ListResourceTypesByProviderResponse.Builder builder = ListResourceTypesByProviderResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ResourcesManagementPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addResourceTypes(toResourceType(e)));
        return builder.build();
    }

    private ListResourceSegmentationsByProviderResponse toPageSegmentation(Page<ResourceSegmentationModel> page) {
        ListResourceSegmentationsByProviderResponse.Builder builder = ListResourceSegmentationsByProviderResponse
                .newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ResourcesManagementPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addResourceSegmentations(toResourceSegmentation(e)));
        return builder.build();
    }

    private ListResourceSegmentsBySegmentationResponse toPageSegment(Page<ResourceSegmentModel> page) {
        ListResourceSegmentsBySegmentationResponse.Builder builder = ListResourceSegmentsBySegmentationResponse
                .newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ResourcesManagementPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addResourceSegments(toResourceSegment(e)));
        return builder.build();
    }

    private ListFullResourcesByProviderResponse toPageResource(Page<ResourceModel> page) {
        ListFullResourcesByProviderResponse.Builder builder = ListFullResourcesByProviderResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ResourcesManagementPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addResources(toResource(e)));
        return builder.build();
    }

    private ListFullAccountsSpacesByProviderResponse toFullAccountsSpaces(Page<AccountSpaceModel> page) {
        ListFullAccountsSpacesByProviderResponse.Builder builder =
                ListFullAccountsSpacesByProviderResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ResourcesManagementPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(a -> builder.addAccountsSpaces(toProto(a)));
        return builder.build();
    }

    private FullAccountsSpace toProto(AccountSpaceModel m) {
        FullAccountsSpace.Builder builder = FullAccountsSpace.newBuilder();
        builder.setAccountsSpaceId(m.getId())
                .setProviderId(m.getProviderId())
                .setVersion(m.getVersion())
                .setNameEn(m.getNameEn())
                .setNameRu(m.getNameRu())
                .setDescriptionEn(m.getDescriptionEn())
                .setDescriptionRu(m.getDescriptionRu())
                .setReadOnly(m.isReadOnly())
                .setSyncEnabled(m.isSyncEnabled());
        if (m.getOuterKeyInProvider() != null) {
            builder.setKey(m.getOuterKeyInProvider());
        }
        m.getSegments().forEach(s -> builder.addSegments(
                ResourceSegmentIds.newBuilder()
                        .setSegmentationId(s.getSegmentationId())
                        .setSegmentId(s.getSegmentId())
        ));
        m.getUiSettings().ifPresent(it -> builder.setUiSettings(converter.toProto(it)));
        return builder.build();
    }

    private FullResourceType toResourceType(ResourceTypeModel resourceType) {
        FullResourceType.Builder builder = FullResourceType.newBuilder();
        builder.setResourceTypeId(resourceType.getId());
        builder.setProviderId(resourceType.getProviderId());
        builder.setVersion(resourceType.getVersion());
        builder.setKey(resourceType.getKey());
        builder.setNameEn(resourceType.getNameEn());
        builder.setNameRu(resourceType.getNameRu());
        builder.setDescriptionEn(resourceType.getDescriptionEn());
        builder.setDescriptionRu(resourceType.getDescriptionRu());
        builder.setUnitsEnsembleId(resourceType.getUnitsEnsembleId());
        if (resourceType.getAggregationSettings().isPresent()) {
            builder.setAggregationSettings(AggregationSettingsHelper.toAggregationSettings(resourceType
                    .getAggregationSettings().orElseThrow()));
        }
        return builder.build();
    }

    private FullResourceSegmentation toResourceSegmentation(ResourceSegmentationModel resourceSegmentation) {
        FullResourceSegmentation.Builder builder = FullResourceSegmentation.newBuilder();
        builder.setResourceSegmentationId(resourceSegmentation.getId());
        builder.setProviderId(resourceSegmentation.getProviderId());
        builder.setVersion(resourceSegmentation.getVersion());
        builder.setKey(resourceSegmentation.getKey());
        builder.setNameEn(resourceSegmentation.getNameEn());
        builder.setNameRu(resourceSegmentation.getNameRu());
        builder.setDescriptionEn(resourceSegmentation.getDescriptionEn());
        builder.setDescriptionRu(resourceSegmentation.getDescriptionRu());
        builder.setGroupingOrder(resourceSegmentation.getGroupingOrder());
        resourceSegmentation.getUiSettings().map(this::toProto).ifPresent(builder::setUiSettings);
        return builder.build();
    }

    private SegmentationUISettings toProto(
            ru.yandex.intranet.d.model.resources.segmentations.SegmentationUISettings model
    ) {
        SegmentationUISettings.Builder build = SegmentationUISettings.newBuilder();
        if (model.getChoiceOrder() != null) {
            build.setChoiceOrder(model.getChoiceOrder());
        }
        if (model.getInSameBlockWithPrevious() != null) {
            build.setInSameBlockWithPrevious(model.getInSameBlockWithPrevious());
        }
        if (model.getTitle() != null) {
            build.setTitle(converter.toProto(model.getTitle()));
        }
        return build.build();
    }

    private FullResourceSegment toResourceSegment(ResourceSegmentModel resourceSegment) {
        FullResourceSegment.Builder builder = FullResourceSegment.newBuilder();
        builder.setResourceSegmentId(resourceSegment.getId());
        builder.setResourceSegmentationId(resourceSegment.getSegmentationId());
        builder.setVersion(resourceSegment.getVersion());
        builder.setKey(resourceSegment.getKey());
        builder.setNameEn(resourceSegment.getNameEn());
        builder.setNameRu(resourceSegment.getNameRu());
        builder.setDescriptionEn(resourceSegment.getDescriptionEn());
        builder.setDescriptionRu(resourceSegment.getDescriptionRu());
        builder.setUncommonSegment(resourceSegment.getUncommon().orElse(false));
        return builder.build();
    }

    private FullResource toResource(ResourceModel resource) {
        FullResource.Builder builder = FullResource.newBuilder();
        builder.setResourceId(resource.getId());
        builder.setResourceKey(resource.getKey());
        builder.setProviderId(resource.getProviderId());
        builder.setUnitsEnsembleId(resource.getUnitsEnsembleId());
        builder.setVersion(resource.getVersion());
        builder.setNameEn(resource.getNameEn());
        builder.setNameRu(resource.getNameRu());
        builder.setDescriptionEn(resource.getDescriptionEn());
        builder.setDescriptionRu(resource.getDescriptionRu());
        builder.setReadOnly(resource.isReadOnly());
        builder.setManaged(resource.isManaged());
        builder.setOrderable(resource.isOrderable());
        builder.addAllAllowedUnitIds(resource.getResourceUnits().getAllowedUnitIds());
        String providerApiUnitId = resource.getResourceUnits().getProviderApiUnitId() == null ? "" :
                resource.getResourceUnits().getProviderApiUnitId();
        builder.setProviderApiUnitId(providerApiUnitId);
        builder.setDefaultUnitId(resource.getResourceUnits().getDefaultUnitId());
        builder.setVirtual(resource.isVirtual());
        builder.setAllocatedSupported(FeatureStateHelper.toFeatureState(resource.isAllocatedSupported()
                .orElse(null)));
        if (resource.getResourceTypeId() != null) {
            ResourceSegmentationsIds.Builder segments = ResourceSegmentationsIds.newBuilder();
            segments.setResourceTypeId(resource.getResourceTypeId());
            if (resource.getSegments() != null) {
                resource.getSegments().forEach(s -> {
                    ResourceSegmentIds.Builder segment = ResourceSegmentIds.newBuilder();
                    segment.setSegmentationId(s.getSegmentationId());
                    segment.setSegmentId(s.getSegmentId());
                    segments.addSegments(segment.build());
                });
            }
            builder.setSegmentations(segments.build());
        }
        if (resource.getAggregationSettings().isPresent()) {
            builder.setAggregationSettings(AggregationSettingsHelper.toAggregationSettings(resource
                    .getAggregationSettings().orElseThrow()));
        }
        return builder.build();
    }

    private Result<NewResourceType> validateCreateRequest(CreateResourceTypeRequest request, Locale locale) {
        if (!request.hasResourceType()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource_type", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResourceType());
    }

    private Result<NewResourceSegmentation> validateCreateRequest(CreateResourceSegmentationRequest request,
                                                                  Locale locale) {
        if (!request.hasResourceSegmentation()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource_segmentation",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResourceSegmentation());
    }

    private Result<NewResourceSegment> validateCreateRequest(CreateResourceSegmentRequest request, Locale locale) {
        if (!request.hasResourceSegment()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource_segment",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResourceSegment());
    }

    private Result<NewResource> validateCreateRequest(CreateResourceRequest request, Locale locale) {
        if (!request.hasResource()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResource());
    }

    private Result<NewAccountsSpace> validateCreateRequest(CreateAccountsSpaceRequest request, Locale locale) {
        if (!request.hasAccountsSpace()) {
            ErrorCollection error = ErrorCollection.builder().addError("accounts_space", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getAccountsSpace());
    }


    private ResourceTypeCreateDto toResourceTypeCreate(NewResourceType resourceType) {
        return new ResourceTypeCreateDto(resourceType.getKey(), resourceType.getNameEn(), resourceType.getNameRu(),
                resourceType.getDescriptionEn(), resourceType.getDescriptionRu(), resourceType.getUnitsEnsembleId(),
                resourceType.getSortingOrder(),
                resourceType.hasAggregationSettings()
                        ? AggregationSettingsHelper.fromAggregationSettings(resourceType.getAggregationSettings())
                        : null);
    }

    private ResourceSegmentationCreateDto toResourceSegmentationCreate(NewResourceSegmentation resourceSegmentation) {
        return new ResourceSegmentationCreateDto(resourceSegmentation.getKey(), resourceSegmentation.getNameEn(),
                resourceSegmentation.getNameRu(), resourceSegmentation.getDescriptionEn(),
                resourceSegmentation.getDescriptionRu(), resourceSegmentation.getGroupingOrder(),
                resourceSegmentation.hasUiSettings() ?
                        new SegmentationUISettingsDto(resourceSegmentation.getUiSettings()) : null
        );
    }

    private ResourceSegmentCreateDto toResourceSegmentCreate(NewResourceSegment resourceSegment) {
        return new ResourceSegmentCreateDto(resourceSegment.getKey(), resourceSegment.getNameEn(),
                resourceSegment.getNameRu(), resourceSegment.getDescriptionEn(),
                resourceSegment.getDescriptionRu(), resourceSegment.getUncommonSegment());
    }

    private ResourceCreateDto toResourceCreate(NewResource resource) {
        return new ResourceCreateDto(resource.getResourceKey(),
                resource.getUnitsEnsembleId(),
                resource.getNameEn(),
                resource.getNameRu(),
                resource.getDescriptionEn(),
                resource.getDescriptionRu(),
                resource.getReadOnly(),
                resource.getOrderable(),
                resource.getManaged(),
                resource.getAllowedUnitIdsList(),
                resource.getDefaultUnitId(),
                resource.getProviderApiUnitId(),
                toResourceTypeSegmentsCreate(resource),
                resource.getDefaultQuota() > 0 ? resource.getDefaultQuota() : null,
                resource.getVirtual(),
                FeatureStateHelper.fromFeatureState(resource.getAllocatedSupported()),
                resource.hasAggregationSettings()
                        ? AggregationSettingsHelper.fromAggregationSettings(resource.getAggregationSettings())
                        : null);
    }

    private AccountsSpaceCreateDto toAccountsSpaceCreateDto(NewAccountsSpace a) {
        return new AccountsSpaceCreateDto(
                a.getKey(),
                a.getNameEn(),
                a.getNameRu(),
                a.getDescriptionEn(),
                a.getDescriptionRu(),
                a.getReadOnly(),
                toCreateResourceSegmentationSegmentDtos(a.getSegmentsList()),
                a.hasUiSettings() ? new ProviderUISettingsDto(a.getUiSettings()) : null,
                a.getSyncEnabled()
        );
    }

    private CreateResourceTypeSegmentsDto toResourceTypeSegmentsCreate(NewResource resource) {
        if (!resource.hasSegmentations()) {
            return null;
        }
        List<CreateResourceSegmentationSegmentDto> segments = toCreateResourceSegmentationSegmentDtos(
                resource.getSegmentations().getSegmentsList()
        );
        return new CreateResourceTypeSegmentsDto(resource.getSegmentations().getResourceTypeId(), segments);
    }

    private List<CreateResourceSegmentationSegmentDto> toCreateResourceSegmentationSegmentDtos(
            List<ResourceSegmentIds> segmentsList
    ) {
        return segmentsList
                .stream().map(v -> new CreateResourceSegmentationSegmentDto(v.getSegmentationId(), v.getSegmentId()))
                .collect(Collectors.toList());
    }

    private Result<UpdatedResourceType> validateUpdateRequest(UpdateResourceTypeRequest request, Locale locale) {
        if (!request.hasResourceType()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource_type", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResourceType());
    }

    private Result<UpdatedResourceType> validatePatchRequest(PatchResourceTypeRequest request, Locale locale) {
        if (!request.hasResourceType()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource_type", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResourceType());
    }

    private Result<UpdatedResourceSegmentation> validateUpdateRequest(UpdateResourceSegmentationRequest request,
                                                                      Locale locale) {
        if (!request.hasResourceSegmentation()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource_segmentation",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResourceSegmentation());
    }

    private Result<UpdatedResourceSegment> validateUpdateRequest(UpdateResourceSegmentRequest request,
                                                                 Locale locale) {
        if (!request.hasResourceSegment()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource_segment",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResourceSegment());
    }

    private Result<UpdatedResource> validateUpdateRequest(UpdateResourceRequest request, Locale locale) {
        if (!request.hasResource()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getResource());
    }

    private Result<UpdatedAccountsSpace> validateUpdateRequest(UpdateAccountsSpaceRequest request, Locale locale) {
        if (!request.hasAccountsSpace()) {
            ErrorCollection error = ErrorCollection.builder().addError("resource", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(request.getAccountsSpace());
    }


    private ResourceTypePutDto toResourceTypeUpdate(UpdatedResourceType resourceType) {
        return new ResourceTypePutDto(resourceType.getNameEn(), resourceType.getNameRu(),
                resourceType.getDescriptionEn(), resourceType.getDescriptionRu(), resourceType.getSortingOrder(),
                resourceType.hasAggregationSettings()
                        ? AggregationSettingsHelper.fromAggregationSettings(resourceType.getAggregationSettings())
                        : null);
    }

    private ResourceTypePatchDto toResourceTypePatch(UpdatedResourceType resourceType, FieldMask fieldMask) {
        Set<String> filedMaskFields = new HashSet<>(fieldMask.getPathsList());
        Set<String> objectFields = resourceType.getAllFields().keySet().stream()
                .map(Descriptors.FieldDescriptor::getFullName)
                .collect(Collectors.toSet());
        PatchedValueGrpcFactory factory = new PatchedValueGrpcFactory(filedMaskFields, objectFields,
                UpdatedResourceType.getDescriptor().getFullName());

        return new ResourceTypePatchDto(
                factory.create(resourceType.getNameEn(), "name_en"),
                factory.create(resourceType.getNameRu(), "name_ru"),
                factory.create(resourceType.getDescriptionEn(), "description_en"),
                factory.create(resourceType.getDescriptionRu(), "description_ru"),
                factory.create(resourceType.getSortingOrder(), "sorting_order"),
                factory.create(resourceType.hasAggregationSettings() ?
                        AggregationSettingsHelper.fromAggregationSettings(resourceType.getAggregationSettings()) : null,
                        "aggregation_settings")
        );
    }

    private ResourceSegmentationPutDto toResourceSegmentationUpdate(
            UpdatedResourceSegmentation resourceSegmentation) {
        return new ResourceSegmentationPutDto(resourceSegmentation.getNameEn(), resourceSegmentation.getNameRu(),
                resourceSegmentation.getDescriptionEn(), resourceSegmentation.getDescriptionRu(),
                resourceSegmentation.getGroupingOrder(),
                resourceSegmentation.hasUiSettings() ?
                        new SegmentationUISettingsDto(resourceSegmentation.getUiSettings()) : null
        );
    }

    private ResourceSegmentPutDto toResourceSegmentUpdate(UpdatedResourceSegment resourceSegment) {
        return new ResourceSegmentPutDto(resourceSegment.getNameEn(), resourceSegment.getNameRu(),
                resourceSegment.getDescriptionEn(), resourceSegment.getDescriptionRu(),
                resourceSegment.getUncommonSegment());
    }

    private ResourcePutDto toResourceUpdate(UpdatedResource resource) {
        return new ResourcePutDto(resource.getNameEn(),
                resource.getNameRu(),
                resource.getDescriptionEn(),
                resource.getDescriptionRu(),
                resource.getOrderable(),
                resource.getManaged(),
                resource.getAllowedUnitIdsList(),
                resource.getDefaultUnitId(),
                resource.getProviderApiUnitId(),
                resource.getDefaultQuota() > 0 ? resource.getDefaultQuota() : null,
                resource.getVirtual(),
                FeatureStateHelper.fromFeatureState(resource.getAllocatedSupported()),
                resource.hasAggregationSettings()
                        ? AggregationSettingsHelper.fromAggregationSettings(resource.getAggregationSettings())
                        : null);
    }

    private AccountsSpacePutDto toAccountsSpacePutDto(UpdatedAccountsSpace a) {
        return new AccountsSpacePutDto(
                a.getNameEn(),
                a.getNameRu(),
                a.getDescriptionEn(),
                a.getDescriptionRu(),
                a.hasUiSettings() ? new ProviderUISettingsDto(a.getUiSettings()) : null,
                a.getSyncEnabled()
        );
    }

}
