package ru.yandex.intranet.d.grpc.services;

import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.AccountsSpace;
import ru.yandex.intranet.d.backend.service.proto.GetResourceRequest;
import ru.yandex.intranet.d.backend.service.proto.ListResourcesByProviderRequest;
import ru.yandex.intranet.d.backend.service.proto.ListResourcesByProviderResponse;
import ru.yandex.intranet.d.backend.service.proto.Resource;
import ru.yandex.intranet.d.backend.service.proto.ResourceSegmentationPair;
import ru.yandex.intranet.d.backend.service.proto.ResourceType;
import ru.yandex.intranet.d.backend.service.proto.ResourcesPageToken;
import ru.yandex.intranet.d.backend.service.proto.ResourcesServiceGrpc;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.resources.ExpandedResources;
import ru.yandex.intranet.d.services.resources.ResourcesService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;


/**
 * GRPC resources service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcResourcesServiceImpl extends ResourcesServiceGrpc.ResourcesServiceImplBase {

    private final ResourcesService resourcesService;
    private final MessageSource messages;
    private final CommonProtoConverter converter;

    public GrpcResourcesServiceImpl(
            ResourcesService resourcesService,
            @Qualifier("messageSource") MessageSource messages,
            CommonProtoConverter converter
    ) {
        this.resourcesService = resourcesService;
        this.messages = messages;
        this.converter = converter;
    }

    @Override
    public void listResourcesByProvider(ListResourcesByProviderRequest request,
                                        StreamObserver<ListResourcesByProviderResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> {
                    boolean withSegmentations = reqParam.hasWithSegmentations()
                            && reqParam.getWithSegmentations().getWithSegmentations();
                    return resourcesService.getPage(reqParam.getProviderId(),
                            toPageRequest(reqParam), currentUser, locale, false, withSegmentations)
                            .flatMap(resp -> resp.match(u -> Mono.just(toPageResource(u, withSegmentations, locale)),
                                    e -> Mono.error(Errors.toGrpcError(e, messages, locale))));
                }), messages);
    }

    @Override
    public void getResource(GetResourceRequest request, StreamObserver<Resource> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> {
                    boolean withSegmentations = reqParam.hasWithSegmentations()
                            && reqParam.getWithSegmentations().getWithSegmentations();
                    return resourcesService.getById(reqParam.getResourceId(),
                            reqParam.getProviderId(), currentUser, locale, false, withSegmentations)
                            .flatMap(resp -> resp.match(u -> Mono.just(toResource(u, withSegmentations, locale)),
                                    e -> Mono.error(Errors.toGrpcError(e, messages, locale))));
                }), messages);
    }

    private PageRequest toPageRequest(ListResourcesByProviderRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private ListResourcesByProviderResponse toPageResource(ExpandedResources<Page<ResourceModel>> page,
                                                           boolean withSegmentations, Locale locale) {
        ListResourcesByProviderResponse.Builder builder = ListResourcesByProviderResponse.newBuilder();
        page.getResources().getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(ResourcesPageToken.newBuilder().setToken(t).build()));
        page.getResources().getItems().forEach(e -> builder.addResources(toResource(e, page.getUnitsEnsembles(),
                page.getResourceTypes(), page.getResourceSegmentations(), page.getResourceSegments(),
                withSegmentations, locale)));
        return builder.build();
    }

    private Resource toResource(ExpandedResources<ResourceModel> expandedResource, boolean withSegmentations,
                                Locale locale) {
        return toResource(expandedResource.getResources(), expandedResource.getUnitsEnsembles(),
                expandedResource.getResourceTypes(), expandedResource.getResourceSegmentations(),
                expandedResource.getResourceSegments(), withSegmentations, locale);
    }

    private Resource toResource(ResourceModel resource,
                                Map<String, UnitsEnsembleModel> unitsEnsembles,
                                Map<String, ResourceTypeModel> resourceTypes,
                                Map<String, ResourceSegmentationModel> resourceSegmentations,
                                Map<String, ResourceSegmentModel> resourceSegments,
                                boolean withSegmentations,
                                Locale locale) {
        Resource.Builder builder = Resource.newBuilder();
        builder.setResourceId(resource.getId());
        builder.setResourceKey(resource.getKey());
        builder.setProviderId(resource.getProviderId());
        builder.setUnitsEnsembleId(resource.getUnitsEnsembleId());
        builder.setVersion(resource.getVersion());
        builder.setName(Locales.select(resource.getNameEn(), resource.getNameRu(), locale));
        builder.setDescription(Locales.select(resource.getDescriptionEn(), resource.getDescriptionRu(), locale));
        builder.setReadOnly(resource.isReadOnly());
        builder.setManaged(resource.isManaged());
        builder.setOrderable(resource.isOrderable());
        builder.addAllAllowedUnitKeys(getAllowedUnitKeys(resource, unitsEnsembles));
        builder.setDefaultUnitKey(getDefaultUnitKey(resource, unitsEnsembles));
        if (withSegmentations && resource.getResourceTypeId() != null) {
            builder.setSegmentations(toSegments(resource.getResourceTypeId(), resource.getSegments(), resourceTypes,
                    resourceSegmentations, resourceSegments, locale));
        }
        if (resource.getAccountsSpacesId() != null) {
            builder.setAccountsSpace(AccountsSpace.newBuilder().setAccountsSpaceId(
                    resource.getAccountsSpacesId()
            ));
        }
        return builder.build();
    }

    private String getDefaultUnitKey(ResourceModel resource,
                                     Map<String, UnitsEnsembleModel> unitsEnsembles) {
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles
                .get(resource.getUnitsEnsembleId());
        return unitsEnsemble.unitById(resource.getResourceUnits().getDefaultUnitId())
                .map(UnitModel::getKey).get();
    }

    private Set<String> getAllowedUnitKeys(ResourceModel resource,
                                           Map<String, UnitsEnsembleModel> unitsEnsembles) {
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        return resource.getResourceUnits().getAllowedUnitIds().stream()
                .map(id -> unitsEnsemble.unitById(id).map(UnitModel::getKey).get())
                .collect(Collectors.toSet());
    }

    private ResourceSegmentationPair toSegments(
            String resourceTypeId,
            Set<ResourceSegmentSettingsModel> segments,
            Map<String, ResourceTypeModel> resourceTypes,
            Map<String, ResourceSegmentationModel> resourceSegmentations,
            Map<String, ResourceSegmentModel> resourceSegments,
            Locale locale
    ) {
        ResourceSegmentationPair.Builder builder = ResourceSegmentationPair.newBuilder();
        ResourceTypeModel resourceType = resourceTypes.get(resourceTypeId);
        builder.setResourceType(toResourceType(resourceType, locale));
        if (segments != null) {
            segments.forEach(s -> {
                ResourceSegmentationModel segmentation = resourceSegmentations.get(s.getSegmentationId());
                ResourceSegmentModel segment = resourceSegments.get(s.getSegmentId());
                builder.addSegments(converter.toSegment(segmentation, segment, locale));
            });
        }
        return builder.build();
    }

    private ResourceType toResourceType(ResourceTypeModel resourceType, Locale locale) {
        ResourceType.Builder builder = ResourceType.newBuilder();
        builder.setResourceTypeId(resourceType.getId());
        builder.setKey(resourceType.getKey());
        builder.setName(Locales.select(resourceType.getNameEn(), resourceType.getNameRu(), locale));
        builder.setDescription(Locales.select(resourceType.getDescriptionEn(),
                resourceType.getDescriptionRu(), locale));
        return builder.build();
    }


}
