package ru.yandex.intranet.d.grpc.services;

import java.util.Locale;
import java.util.Optional;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.CancelTransferRequest;
import ru.yandex.intranet.d.backend.service.proto.CreateTransferRequest;
import ru.yandex.intranet.d.backend.service.proto.GetTransferRequest;
import ru.yandex.intranet.d.backend.service.proto.Transfer;
import ru.yandex.intranet.d.backend.service.proto.TransfersServiceGrpc;
import ru.yandex.intranet.d.backend.service.proto.VoteForTransferRequest;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.grpc.GrpcIdempotency;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.services.transfer.TransferRequestLogicService;
import ru.yandex.intranet.d.services.transfer.TransferRequestService;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC accounts service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcTransfersServiceImpl extends TransfersServiceGrpc.TransfersServiceImplBase {

    private final TransferRequestService transferRequestService;
    private final TransferRequestLogicService transferRequestLogicService;
    private final MessageSource messages;

    public GrpcTransfersServiceImpl(TransferRequestService transferRequestService,
                                    TransferRequestLogicService transferRequestLogicService,
                                    @Qualifier("messageSource") MessageSource messages) {
        this.transferRequestService = transferRequestService;
        this.transferRequestLogicService = transferRequestLogicService;
        this.messages = messages;
    }

    @Override
    public void getTransfer(GetTransferRequest request, StreamObserver<Transfer> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> transferRequestService.getOne(reqParam.getTransferId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(TransferDtoGrpcMapping.toTransfer(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createTransfer(CreateTransferRequest request, StreamObserver<Transfer> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Optional<String> idempotencyKey = GrpcIdempotency.idempotencyKey();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> transferRequestLogicService.createMono(
                        TransferDtoGrpcMapping.toCreateRequest(reqParam),
                                currentUser, locale, true, idempotencyKey.orElse(null),
                                request.getDelayValidation())
                        .flatMap(resp -> resp.match(u -> Mono.just(TransferDtoGrpcMapping.toTransfer(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void voteForTransfer(VoteForTransferRequest request, StreamObserver<Transfer> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Optional<String> idempotencyKey = GrpcIdempotency.idempotencyKey();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> transferRequestLogicService.voteMono(reqParam.getTransferId(),
                                reqParam.getVersion(), TransferDtoGrpcMapping.toVoteParameters(request), currentUser,
                                locale, idempotencyKey.orElse(null))
                        .flatMap(resp -> resp.match(u -> Mono.just(TransferDtoGrpcMapping.toTransfer(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void cancelTransfer(CancelTransferRequest request, StreamObserver<Transfer> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Optional<String> idempotencyKey = GrpcIdempotency.idempotencyKey();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> transferRequestLogicService.cancelMono(reqParam.getTransferId(),
                                reqParam.getVersion(), currentUser, locale, idempotencyKey.orElse(null))
                        .flatMap(resp -> resp.match(u -> Mono.just(TransferDtoGrpcMapping.toTransfer(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

}
