package ru.yandex.intranet.d.grpc.services;

import java.util.Locale;

import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.proto.GetUnitRequest;
import ru.yandex.intranet.d.backend.service.proto.GetUnitsEnsembleRequest;
import ru.yandex.intranet.d.backend.service.proto.ListUnitsEnsemblesRequest;
import ru.yandex.intranet.d.backend.service.proto.ListUnitsEnsemblesResponse;
import ru.yandex.intranet.d.backend.service.proto.Unit;
import ru.yandex.intranet.d.backend.service.proto.UnitsEnsemble;
import ru.yandex.intranet.d.backend.service.proto.UnitsPageToken;
import ru.yandex.intranet.d.backend.service.proto.UnitsServiceGrpc;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.services.units.UnitsEnsemblesService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC units service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
public class GrpcUnitsServiceImpl extends UnitsServiceGrpc.UnitsServiceImplBase {

    private final UnitsEnsemblesService unitsEnsemblesService;
    private final MessageSource messages;

    public GrpcUnitsServiceImpl(UnitsEnsemblesService unitsEnsemblesService,
                                @Qualifier("messageSource") MessageSource messages) {
        this.unitsEnsemblesService = unitsEnsemblesService;
        this.messages = messages;
    }

    @Override
    public void listUnitsEnsembles(ListUnitsEnsemblesRequest request,
                                      StreamObserver<ListUnitsEnsemblesResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> unitsEnsemblesService.getPage(toPageRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toPage(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getUnitsEnsemble(GetUnitsEnsembleRequest request,
                                 StreamObserver<UnitsEnsemble> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> unitsEnsemblesService.getById(reqParam.getUnitsEnsembleId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toUnitsEnsemble(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getUnit(GetUnitRequest request, StreamObserver<Unit> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> unitsEnsemblesService.getUnitById(reqParam.getUnitsEnsembleId(),
                        reqParam.getUnitId(), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toUnit(u, locale)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private PageRequest toPageRequest(ListUnitsEnsemblesRequest request) {
        String continuationToken = request.hasPageToken()
                ? request.getPageToken().getToken() : null;
        Long limit = request.hasLimit() ? request.getLimit().getLimit() : null;
        return new PageRequest(continuationToken, limit);
    }

    private ListUnitsEnsemblesResponse toPage(Page<UnitsEnsembleModel> page, Locale locale) {
        ListUnitsEnsemblesResponse.Builder builder = ListUnitsEnsemblesResponse.newBuilder();
        page.getContinuationToken().ifPresent(t -> builder
                .setNextPageToken(UnitsPageToken.newBuilder().setToken(t).build()));
        page.getItems().forEach(e -> builder.addUnitsEnsembles(toUnitsEnsemble(e, locale)));
        return builder.build();
    }

    private UnitsEnsemble toUnitsEnsemble(UnitsEnsembleModel unitsEnsemble, Locale locale) {
        UnitsEnsemble.Builder builder = UnitsEnsemble.newBuilder();
        builder.setUnitsEnsembleId(unitsEnsemble.getId());
        builder.setVersion(unitsEnsemble.getVersion());
        builder.setName(Locales.select(unitsEnsemble.getNameEn(), unitsEnsemble.getNameRu(), locale));
        builder.setDescription(Locales.select(unitsEnsemble.getDescriptionEn(),
                unitsEnsemble.getDescriptionRu(), locale));
        builder.setFractionsAllowed(unitsEnsemble.isFractionsAllowed());
        unitsEnsemble.getUnits().stream().sorted(UnitsComparator.INSTANCE)
                .forEach(u -> builder.addUnits(toUnit(u, locale)));
        builder.setKey(unitsEnsemble.getKey());
        return builder.build();
    }

    private Unit toUnit(UnitModel unit, Locale locale) {
        return Unit.newBuilder()
                .setUnitId(unit.getId())
                .setUnitKey(unit.getKey())
                .setShortName(Locales.select(unit.getShortNameSingularEn(),
                        unit.getShortNameSingularRu().get(GrammaticalCase.NOMINATIVE), locale))
                .setLongName(Locales.select(unit.getLongNameSingularEn(),
                        unit.getLongNameSingularRu().get(GrammaticalCase.NOMINATIVE), locale))
                .setBase(unit.getBase())
                .setPower(unit.getPower())
                .build();
    }

}
