package ru.yandex.intranet.d.grpc.services;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Maps;
import com.google.protobuf.util.Timestamps;
import org.jetbrains.annotations.NotNull;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.backend.service.proto.CreateTransferRequest;
import ru.yandex.intranet.d.backend.service.proto.GroupedTransferResponsible;
import ru.yandex.intranet.d.backend.service.proto.GroupedTransferResponsibleUser;
import ru.yandex.intranet.d.backend.service.proto.ProviderReserveResponsible;
import ru.yandex.intranet.d.backend.service.proto.ProviderSuperResponsible;
import ru.yandex.intranet.d.backend.service.proto.ProvisionTransfer;
import ru.yandex.intranet.d.backend.service.proto.ProvisionTransferParameters;
import ru.yandex.intranet.d.backend.service.proto.QuotaTransfer;
import ru.yandex.intranet.d.backend.service.proto.QuotaTransferParameters;
import ru.yandex.intranet.d.backend.service.proto.ReserveResourceTransfer;
import ru.yandex.intranet.d.backend.service.proto.ReserveTransferParameters;
import ru.yandex.intranet.d.backend.service.proto.ResourceTransfer;
import ru.yandex.intranet.d.backend.service.proto.Transfer;
import ru.yandex.intranet.d.backend.service.proto.TransferAmount;
import ru.yandex.intranet.d.backend.service.proto.TransferApplication;
import ru.yandex.intranet.d.backend.service.proto.TransferDescription;
import ru.yandex.intranet.d.backend.service.proto.TransferErrorDetails;
import ru.yandex.intranet.d.backend.service.proto.TransferFailures;
import ru.yandex.intranet.d.backend.service.proto.TransferFieldError;
import ru.yandex.intranet.d.backend.service.proto.TransferLoanBorrowMeta;
import ru.yandex.intranet.d.backend.service.proto.TransferLoanMeta;
import ru.yandex.intranet.d.backend.service.proto.TransferLoanParameters;
import ru.yandex.intranet.d.backend.service.proto.TransferLoanPayOffMeta;
import ru.yandex.intranet.d.backend.service.proto.TransferOperation;
import ru.yandex.intranet.d.backend.service.proto.TransferParameters;
import ru.yandex.intranet.d.backend.service.proto.TransferResponsible;
import ru.yandex.intranet.d.backend.service.proto.TransferStatus;
import ru.yandex.intranet.d.backend.service.proto.TransferSubtype;
import ru.yandex.intranet.d.backend.service.proto.TransferTrackerIssue;
import ru.yandex.intranet.d.backend.service.proto.TransferType;
import ru.yandex.intranet.d.backend.service.proto.TransferUser;
import ru.yandex.intranet.d.backend.service.proto.TransferVoter;
import ru.yandex.intranet.d.backend.service.proto.TransferVotes;
import ru.yandex.intranet.d.backend.service.proto.VoteForTransferRequest;
import ru.yandex.intranet.d.backend.service.proto.VoteType;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.transfers.FoldersResponsible;
import ru.yandex.intranet.d.model.transfers.LoanMeta;
import ru.yandex.intranet.d.model.transfers.LocalizedTransferApplicationErrors;
import ru.yandex.intranet.d.model.transfers.OperationStatus;
import ru.yandex.intranet.d.model.transfers.ProviderResponsible;
import ru.yandex.intranet.d.model.transfers.ReserveResponsibleModel;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.TransferApplicationDetails;
import ru.yandex.intranet.d.model.transfers.TransferApplicationErrors;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.transfers.TransferRequestSubtype;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.util.JsonUtil;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestSubtypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestVoteTypeDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateProvisionTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaResourceTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateReserveTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanBorrowParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanPayOffParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVotingDto;

/**
 * Transfer DTO GRPC mapping.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferDtoGrpcMapping {

    private TransferDtoGrpcMapping() {
    }

    public static FrontTransferRequestVotingDto toVoteParameters(VoteForTransferRequest request) {
        return new FrontTransferRequestVotingDto(fromVoteType(request.getVote()));
    }

    public static FrontCreateTransferRequestDto toCreateRequest(CreateTransferRequest request) {
        FrontCreateTransferRequestDto.Builder result = FrontCreateTransferRequestDto.builder();
        if (request.hasDescription()) {
            result.description(request.getDescription().getText());
        }
        result.addConfirmation(request.getAddConfirmation());
        result.requestType(fromRequestType(request.getType()));
        if (request.hasParameters()) {
            result.parameters(toCreateParameters(request.getParameters()));
        }
        if (request.hasLoanParameters()) {
            result.loanParameters(toLoanParameters(request.getLoanParameters()));
        }
        return result.build();
    }

    public static Transfer toTransfer(ExpandedTransferRequests<TransferRequestModel> transfer, Locale locale) {
        Transfer.Builder builder = Transfer.newBuilder();
        builder.setTransferId(transfer.getTransferRequests().getId());
        builder.setVersion(transfer.getTransferRequests().getVersion());
        transfer.getTransferRequests().getDescription()
                .ifPresent(v -> builder.setDescription(TransferDescription.newBuilder().setText(v).build()));
        transfer.getTransferRequests().getTrackerIssueKey()
                .ifPresent(v -> builder.setTrackerIssue(TransferTrackerIssue.newBuilder().setKey(v).build()));
        builder.setType(toType(transfer.getTransferRequests().getType()));
        transfer.getTransferRequests().getSubtype()
                .ifPresent(subtype -> builder.setTransferSubtype(toSubtype(subtype)));
        builder.setStatus(toStatus(transfer.getTransferRequests().getStatus()));
        builder.setCreatedBy(TransferUser.newBuilder().setUid(transfer.getUsers()
                .get(transfer.getTransferRequests().getCreatedBy()).getPassportUid().orElse("")).build());
        transfer.getTransferRequests().getUpdatedBy().ifPresent(v -> builder.setUpdatedBy(TransferUser.newBuilder()
                .setUid(transfer.getUsers().get(v).getPassportUid().orElse("")).build()));
        builder.setCreatedAt(Timestamps.fromMillis(transfer.getTransferRequests().getCreatedAt().toEpochMilli()));
        transfer.getTransferRequests().getUpdatedAt()
                .ifPresent(v -> builder.setUpdatedAt(Timestamps.fromMillis(v.toEpochMilli())));
        transfer.getTransferRequests().getAppliedAt()
                .ifPresent(v -> builder.setAppliedAt(Timestamps.fromMillis(v.toEpochMilli())));
        builder.setParameters(toParameters(transfer.getTransferRequests().getParameters(),
                transfer.getTransferRequests().getType(), transfer.getResources(), transfer.getUnitsEnsembles(),
                transfer.getFolders()));
        builder.setResponsible(toResponsible(transfer.getTransferRequests().getResponsible(), transfer.getUsers()));
        builder.setVotes(toVotes(transfer.getTransferRequests().getVotes(), transfer.getUsers()));
        transfer.getTransferRequests().getApplicationDetails()
                .ifPresent(v -> builder.setApplication(toApplication(v, locale)));
        transfer.getTransferRequests().getLoanMeta().ifPresent(v -> builder.setLoanMeta(toLoanMeta(v)));
        return builder.build();
    }

    private static TransferLoanMeta toLoanMeta(LoanMeta loanMeta) {
        TransferLoanMeta.Builder result = TransferLoanMeta.newBuilder();
        switch (loanMeta.getOperationType()) {
            case BORROW -> {
                TransferLoanBorrowMeta.Builder borrowResult = TransferLoanBorrowMeta.newBuilder();
                LocalDate dueDate = Objects.requireNonNull(loanMeta.getBorrowDueDate()).getLocalDate();
                borrowResult.setDueDate(ru.yandex.intranet.d.backend.service.proto.LocalDate.newBuilder()
                                .setYear(dueDate.getYear())
                                .setMonth(dueDate.getMonthValue())
                                .setDay(dueDate.getDayOfMonth())
                                .build());
                if (loanMeta.getBorrowLoanIds() != null) {
                    borrowResult.addAllLoanIds(loanMeta.getBorrowLoanIds());
                }
                result.setBorrowMeta(borrowResult.build());
            }
            case PAY_OFF -> {
                TransferLoanPayOffMeta.Builder payOffBuilder = TransferLoanPayOffMeta.newBuilder();
                payOffBuilder.setLoanId(Objects.requireNonNull(loanMeta.getPayOffLoanId()));
                result.setPayOffMeta(payOffBuilder.build());
            }
            default -> {
            }
        }
        result.setProvideOverCommitReserve(Boolean.TRUE.equals(loanMeta.getProvideOverCommitReserve()));
        return result.build();
    }

    private static FrontCreateTransferRequestParametersDto toCreateParameters(TransferParameters parameters) {
        FrontCreateTransferRequestParametersDto.Builder result = FrontCreateTransferRequestParametersDto.builder();
        if (parameters.hasQuotaTransfer()) {
            parameters.getQuotaTransfer().getQuotaTransfersList().forEach(transfer ->
                    result.addQuotaTransfer(toCreateQuotaTransfer(transfer)));
        }
        if (parameters.hasReserveTransfer()) {
            result.reserveTransfer(toCreateReserveTransfer(parameters.getReserveTransfer()));
        }
        if (parameters.hasProvisionTransfer()) {
            parameters.getProvisionTransfer().getProvisionTransferList().forEach(provisionTransfer ->
                    result.addProvisionTransfer(toProvisionTransfer(provisionTransfer)));
        }
        return result.build();
    }

    private static FrontTransferLoanParametersDto toLoanParameters(TransferLoanParameters loanParameters) {
        FrontTransferLoanBorrowParametersDto borrowParameters = null;
        FrontTransferLoanPayOffParametersDto payOffParameters = null;
        if (loanParameters.hasBorrowParameters()) {
            ru.yandex.intranet.d.backend.service.proto.LocalDate dueDate = loanParameters
                    .getBorrowParameters().getDueDate();
            borrowParameters = new FrontTransferLoanBorrowParametersDto(LocalDate.of(dueDate.getYear(),
                    dueDate.getMonth(), dueDate.getDay()));
        }
        if (loanParameters.hasPayOffParameters()) {
            payOffParameters = new FrontTransferLoanPayOffParametersDto(loanParameters
                    .getPayOffParameters().getLoanId());
        }
        return new FrontTransferLoanParametersDto(borrowParameters, payOffParameters,
                loanParameters.getProvideOverCommitReserve());
    }

    private static FrontCreateQuotaTransferDto toCreateQuotaTransfer(QuotaTransfer transfer) {
        FrontCreateQuotaTransferDto.Builder result = FrontCreateQuotaTransferDto.builder();
        result.destinationFolderId(transfer.getFolderId());
        // Not nice but... service id is not present here, validator should behave differently depending on endpoint
        transfer.getResourceTransfersList().forEach(resourceTransfer -> {
            FrontCreateQuotaResourceTransferDto.Builder builder = FrontCreateQuotaResourceTransferDto.builder();
            builder.resourceId(resourceTransfer.getResourceId());
            // Not nice but... incoming provider id is ignored here, it isn't strictly necessary anyway...
            if (resourceTransfer.hasDelta()) {
                builder.delta(String.valueOf(resourceTransfer.getDelta().getValue()));
                // Not nice but... this should be interpreted differently by validator depending on endpoint
                builder.deltaUnitId(resourceTransfer.getDelta().getUnitKey());
            }
            result.addResourceTransfer(builder.build());
        });
        return result.build();
    }

    private static FrontCreateProvisionTransferDto toProvisionTransfer(ProvisionTransfer transfer) {
        return new FrontCreateProvisionTransferDto(
                transfer.getSourceAccountId(),
                transfer.getSourceFolderId(),
                null,
                transfer.getDestinationAccountId(),
                transfer.getDestinationFolderId(),
                null,
                toQuotaResourceTransfers(transfer.getSourceAccountTransfersList()),
                toQuotaResourceTransfers(transfer.getDestinationAccountTransfersList())
        );
    }

    private static ProvisionTransfer toProvisionTransfer(
            ru.yandex.intranet.d.model.transfers.ProvisionTransfer provisionTransfer,
            Map<String, ResourceModel> resources,
            Map<String, UnitsEnsembleModel> unitsEnsembles
    ) {
        ProvisionTransfer.Builder builder = ProvisionTransfer.newBuilder()
                .setSourceAccountId(provisionTransfer.getSourceAccountId())
                .setDestinationAccountId(provisionTransfer.getDestinationAccountId())
                .setSourceFolderId(provisionTransfer.getSourceFolderId())
                .setDestinationFolderId(provisionTransfer.getDestinationFolderId())
                .addAllSourceAccountTransfers(
                        provisionTransfer.getSourceAccountTransfers().stream().map(resourceQuotaTransfer ->
                                toResourceTransfer(resourceQuotaTransfer, resources, unitsEnsembles)
                        ).collect(Collectors.toList())
                )
                .addAllDestinationAccountTransfers(
                        provisionTransfer.getDestinationAccountTransfers().stream().map(resourceQuotaTransfer ->
                                toResourceTransfer(resourceQuotaTransfer, resources, unitsEnsembles)
                        ).collect(Collectors.toList())
                );
        if (provisionTransfer.getOperationId() != null) {
            builder.setOperation(TransferOperation.newBuilder()
                    .setOperationId(provisionTransfer.getOperationId())
                    .build());
        }
        return builder.build();
    }

    private static List<FrontCreateQuotaResourceTransferDto> toQuotaResourceTransfers(
            @Nullable List<ResourceTransfer> transfers
    ) {
        if (transfers == null) {
            return List.of();
        }
        return transfers.stream()
                .map(TransferDtoGrpcMapping::toFrontCreateQuotaResourceTransfer)
                .collect(Collectors.toList());
    }

    public static FrontCreateQuotaResourceTransferDto toFrontCreateQuotaResourceTransfer(
            ResourceTransfer resourceTransfer
    ) {
        FrontCreateQuotaResourceTransferDto.Builder builder =
                FrontCreateQuotaResourceTransferDto.builder();
        builder.resourceId(resourceTransfer.getResourceId());
        if (resourceTransfer.hasDelta()) {
            builder.delta(String.valueOf(resourceTransfer.getDelta().getValue()));
            // Not nice but... this should be interpreted differently by validator depending on endpoint
            builder.deltaUnitId(resourceTransfer.getDelta().getUnitKey());
        }
        return builder.build();
    }

    private static FrontCreateReserveTransferDto toCreateReserveTransfer(ReserveTransferParameters transfer) {
        FrontCreateReserveTransferDto.Builder result = FrontCreateReserveTransferDto.builder();
        result.providerId(transfer.getProviderId());
        result.destinationFolderId(transfer.getFolderId());
        // Not nice but... service id is not present here, validator should behave differently depending on endpoint
        transfer.getResourceTransfersList().forEach(resourceTransfer -> {
            FrontCreateQuotaResourceTransferDto.Builder builder = FrontCreateQuotaResourceTransferDto.builder();
            builder.resourceId(resourceTransfer.getResourceId());
            if (resourceTransfer.hasDelta()) {
                builder.delta(String.valueOf(resourceTransfer.getDelta().getValue()));
                // Not nice but... this should be interpreted differently by validator depending on endpoint
                builder.deltaUnitId(resourceTransfer.getDelta().getUnitKey());
            }
            result.addResourceTransfer(builder.build());
        });
        return result.build();
    }

    private static TransferParameters toParameters(ru.yandex.intranet.d.model.transfers.TransferParameters parameters,
                                                   TransferRequestType type,
                                                   Map<String, ResourceModel> resources,
                                                   Map<String, UnitsEnsembleModel> unitsEnsembles,
                                                   Map<String, FolderModel> folders) {
        TransferParameters.Builder result = TransferParameters.newBuilder();
        switch (type) {
            case QUOTA_TRANSFER -> result.setQuotaTransfer(toQuotaTransferParameters(parameters.getQuotaTransfers(),
                    resources, unitsEnsembles));
            case RESERVE_TRANSFER -> result.setReserveTransfer(toReserveTransferParameters(
                    parameters.getQuotaTransfers(), folders, resources, unitsEnsembles));
            case PROVISION_TRANSFER -> result.setProvisionTransfer(
                    toProvisionTransferParameters(parameters.getProvisionTransfers(), resources, unitsEnsembles));
            default -> throw new IllegalArgumentException("Unsupported transfer type: " + type);
        }
        return result.build();
    }

    private static ProvisionTransferParameters toProvisionTransferParameters(
            Set<ru.yandex.intranet.d.model.transfers.ProvisionTransfer> provisionTransfers,
            Map<String, ResourceModel> resources,
            Map<String, UnitsEnsembleModel> unitsEnsembles
    ) {
        List<ProvisionTransfer> collect = provisionTransfers.stream()
                .map(p -> toProvisionTransfer(p, resources, unitsEnsembles))
                .collect(Collectors.toList());
        return ProvisionTransferParameters.newBuilder().addAllProvisionTransfer(collect).build();
    }

    private static QuotaTransferParameters toQuotaTransferParameters(
            Set<ru.yandex.intranet.d.model.transfers.QuotaTransfer> transfers,
            Map<String, ResourceModel> resources, Map<String, UnitsEnsembleModel> unitsEnsembles) {
        QuotaTransferParameters.Builder result = QuotaTransferParameters.newBuilder();
        transfers.forEach(transfer -> result.addQuotaTransfers(toQuotaTransfer(transfer, resources, unitsEnsembles)));
        return result.build();
    }

    private static QuotaTransfer toQuotaTransfer(ru.yandex.intranet.d.model.transfers.QuotaTransfer transfer,
                                                 Map<String, ResourceModel> resources,
                                                 Map<String, UnitsEnsembleModel> unitsEnsembles) {
        QuotaTransfer.Builder result = QuotaTransfer.newBuilder();
        result.setFolderId(transfer.getDestinationFolderId());
        transfer.getTransfers().forEach(resourceTransfer -> result
                .addResourceTransfers(toResourceTransfer(resourceTransfer, resources, unitsEnsembles)));
        return result.build();
    }

    private static ResourceTransfer toResourceTransfer(ResourceQuotaTransfer transfer,
                                                       Map<String, ResourceModel> resources,
                                                       Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ResourceModel resource = resources.get(transfer.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedDelta = Units.convertToApi(transfer.getDelta(), resource, unitsEnsemble);
        ResourceTransfer.Builder result = ResourceTransfer.newBuilder();
        result.setResourceId(transfer.getResourceId());
        result.setProviderId(resource.getProviderId());
        result.setDelta(TransferAmount.newBuilder().setValue(convertedDelta.getT1().longValueExact())
                .setUnitKey(convertedDelta.getT2().getKey()).build());
        return result.build();
    }

    private static ReserveTransferParameters toReserveTransferParameters(
            Set<ru.yandex.intranet.d.model.transfers.QuotaTransfer> transfers,
            Map<String, FolderModel> folders,
            Map<String, ResourceModel> resources,
            Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ReserveTransferParameters.Builder result = ReserveTransferParameters.newBuilder();
        String destinationFolderId = getDestinationFolderId(transfers, folders);
        String sourceProviderId = getSourceProviderId(transfers, resources);
        result.setFolderId(destinationFolderId);
        result.setProviderId(sourceProviderId);
        transfers.stream().filter(t -> t.getDestinationFolderId().equals(destinationFolderId))
                .forEach(t -> t.getTransfers().forEach(r -> result.addResourceTransfers(toReserveResourceTransfer(r,
                        resources, unitsEnsembles))));
        return result.build();
    }

    private static ReserveResourceTransfer toReserveResourceTransfer(ResourceQuotaTransfer transfer,
                                                                     Map<String, ResourceModel> resources,
                                                                     Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ResourceModel resource = resources.get(transfer.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedDelta = Units.convertToApi(transfer.getDelta(), resource, unitsEnsemble);
        ReserveResourceTransfer.Builder result = ReserveResourceTransfer.newBuilder();
        result.setResourceId(transfer.getResourceId());
        result.setDelta(TransferAmount.newBuilder().setValue(convertedDelta.getT1().longValueExact())
                .setUnitKey(convertedDelta.getT2().getKey()).build());
        return result.build();
    }

    private static String getDestinationFolderId(Set<ru.yandex.intranet.d.model.transfers.QuotaTransfer> transfers,
                                                 Map<String, FolderModel> folders) {
        return transfers.stream().filter(transfer -> folders
                        .get(transfer.getDestinationFolderId()).getFolderType() != FolderType.PROVIDER_RESERVE)
                .findFirst().map(ru.yandex.intranet.d.model.transfers.QuotaTransfer::getDestinationFolderId)
                .orElseThrow();
    }

    private static String getSourceProviderId(Set<ru.yandex.intranet.d.model.transfers.QuotaTransfer> transfers,
                                              Map<String, ResourceModel> resources) {
        Set<String> providerIds = new HashSet<>();
        transfers.forEach(t -> t.getTransfers()
                .forEach(r -> providerIds.add(resources.get(r.getResourceId()).getProviderId())));
        return providerIds.stream().findFirst().orElseThrow();
    }

    private static TransferResponsible toResponsible(
            ru.yandex.intranet.d.model.transfers.TransferResponsible responsible, Map<String, UserModel> users) {
        TransferResponsible.Builder result = TransferResponsible.newBuilder();
        responsible.getResponsible().forEach(r -> result.addGrouped(toGroupedTransferResponsible(r, users)));
        responsible.getProviderResponsible()
                .forEach(r -> result.addProviderSuperResponsible(toProviderSuperResponsible(r, users)));
        responsible.getReserveResponsibleModel()
                .ifPresent(r -> result.addProviderReserveResponsible(toProviderReserveResponsible(r, users)));
        return result.build();
    }

    private static GroupedTransferResponsible toGroupedTransferResponsible(FoldersResponsible foldersResponsible,
                                                                           Map<String, UserModel> users) {
        GroupedTransferResponsible.Builder result = GroupedTransferResponsible.newBuilder();
        result.addAllFolderIds(foldersResponsible.getFolderIds());
        Map<String, Set<Long>> userServices = new HashMap<>();
        foldersResponsible.getResponsible().forEach(serviceResponsible -> serviceResponsible.getResponsibleIds()
                .forEach(userId -> userServices.computeIfAbsent(userId, k -> new HashSet<>())
                        .add(serviceResponsible.getServiceId())));
        userServices.forEach((userId, serviceIds) -> {
            GroupedTransferResponsibleUser.Builder responsibleBuilder = GroupedTransferResponsibleUser.newBuilder();
            responsibleBuilder.setResponsible(TransferUser.newBuilder().setUid(users.get(userId)
                    .getPassportUid().orElse("")).build());
            responsibleBuilder.addAllServiceIds(serviceIds);
            result.addResponsibleSet(responsibleBuilder.build());
        });
        return result.build();
    }

    private static ProviderSuperResponsible toProviderSuperResponsible(ProviderResponsible responsible,
                                                                       Map<String, UserModel> users) {
        ProviderSuperResponsible.Builder result = ProviderSuperResponsible.newBuilder();
        result.setResponsible(TransferUser.newBuilder().setUid(users.get(responsible.getResponsibleId())
                .getPassportUid().orElse("")).build());
        result.addAllProviderIds(responsible.getProviderIds());
        return result.build();
    }

    private static ProviderReserveResponsible toProviderReserveResponsible(ReserveResponsibleModel responsible,
                                                                           Map<String, UserModel> users) {
        ProviderReserveResponsible.Builder result = ProviderReserveResponsible.newBuilder();
        result.addAllResponsibleUsers(responsible.getResponsibleIds().stream()
                .map(v -> TransferUser.newBuilder().setUid(users.get(v).getPassportUid().orElse(""))
                        .build()).collect(Collectors.toList()));
        result.setProviderId(responsible.getProviderId());
        result.setFolderId(responsible.getFolderId());
        result.setServiceId(responsible.getServiceId());
        return result.build();
    }

    private static TransferVotes toVotes(ru.yandex.intranet.d.model.transfers.TransferVotes votes,
                                         Map<String, UserModel> users) {
        TransferVotes.Builder result = TransferVotes.newBuilder();
        votes.getVotes().forEach(vote -> {
            TransferVoter.Builder voter = TransferVoter.newBuilder();
            voter.setVoter(TransferUser.newBuilder().setUid(users.get(vote.getUserId()).getPassportUid()
                    .orElse("")).build());
            voter.addAllFolderIds(vote.getFolderIds());
            voter.addAllServiceIds(vote.getServiceIds());
            voter.setTimestamp(Timestamps.fromMillis(vote.getTimestamp().toEpochMilli()));
            voter.setVoteType(toVoteType(vote.getType()));
            voter.addAllProviderIds(vote.getProviderIds());
            result.addVoters(voter.build());
        });
        return result.build();
    }

    private static TransferApplication toApplication(TransferApplicationDetails applicationDetails, Locale locale) {
        TransferApplication.Builder result = TransferApplication.newBuilder();
        Optional<TransferApplicationErrors> errorsO = Locales.selectObject(applicationDetails.getErrorsEn(),
                applicationDetails.getErrorsRu(), locale);
        if (errorsO.isPresent()) {
            TransferApplicationErrors errors = errorsO.get();
            result.setFailures(toFailures(errors));
        }
        result.putAllFailuresByOperationId(Maps.transformValues(applicationDetails.getErrorsByOperationId(),
                e -> toFailures(e, locale)));
        result.putAllOperationStatusById(Maps.transformValues(applicationDetails.getOperationStatusById(),
                TransferDtoGrpcMapping::toOperationStatus));
        return result.build();
    }

    private static TransferFailures toFailures(LocalizedTransferApplicationErrors errors, Locale locale) {
        TransferApplicationErrors transferApplicationErrors = Locales.selectObject(errors.getErrorsEn(),
                errors.getErrorsRu(), locale);
        return toFailures(transferApplicationErrors);
    }

    @NotNull
    private static TransferFailures toFailures(TransferApplicationErrors errors) {
        TransferFailures.Builder failures = TransferFailures.newBuilder();
        failures.addAllErrors(errors.getErrors());
        errors.getFieldErrors().forEach((k, v) -> failures.addFieldErrors(TransferFieldError.newBuilder()
                .setKey(k)
                .addAllErrors(v)
                .build()));
        errors.getDetails().forEach((k, v) -> failures.addDetails(TransferErrorDetails.newBuilder()
                .setKey(k)
                .addAllDetails(v.stream().map(JsonUtil::writeAsString).toList())
                .build()));
        return failures.build();
    }

    private static TransferApplication.OperationStatus toOperationStatus(OperationStatus operationStatus) {
        return switch (operationStatus) {
            case COMPLETED -> TransferApplication.OperationStatus.COMPLETED;
            case FAILED -> TransferApplication.OperationStatus.FAILED;
            case EXECUTING -> TransferApplication.OperationStatus.EXECUTING;
            case UNKNOWN -> TransferApplication.OperationStatus.UNKNOWN;
        };
    }

    private static TransferType toType(TransferRequestType type) {
        return switch (type) {
            case QUOTA_TRANSFER -> TransferType.QUOTA_TRANSFER;
            case RESERVE_TRANSFER -> TransferType.RESERVE_TRANSFER;
            case PROVISION_TRANSFER -> TransferType.PROVISION_TRANSFER;
            default -> throw new IllegalArgumentException("Unsupported transfer type: " + type);
        };
    }

    private static TransferSubtype toSubtype(TransferRequestSubtype subtype) {
        return switch (subtype) {
            case DEFAULT_QUOTA_TRANSFER -> TransferSubtype.DEFAULT_QUOTA_TRANSFER;
            case DEFAULT_RESERVE_TRANSFER -> TransferSubtype.DEFAULT_RESERVE_TRANSFER;
            case DEFAULT_PROVISION_TRANSFER -> TransferSubtype.DEFAULT_PROVISION_TRANSFER;
            case EXCHANGE_PROVISION_TRANSFER -> TransferSubtype.EXCHANGE_PROVISION_TRANSFER;
            case LOAN_PROVISION_TRANSFER -> TransferSubtype.LOAN_PROVISION_TRANSFER;
            case RESERVE_PROVISION_TRANSFER -> TransferSubtype.RESERVE_PROVISION_TRANSFER;
            case DEFAULT_FOLDER_TRANSFER, DEFAULT_ACCOUNT_TRANSFER ->
                    throw new IllegalArgumentException("Unsupported transfer subtype: " + subtype);
        };
    }

    private static TransferStatus toStatus(TransferRequestStatus status) {
        return switch (status) {
            case PENDING -> TransferStatus.PENDING;
            case APPLIED -> TransferStatus.APPLIED;
            case REJECTED -> TransferStatus.REJECTED;
            case CANCELLED -> TransferStatus.CANCELLED;
            case FAILED -> TransferStatus.FAILED;
            case STALE -> TransferStatus.STALE;
            case EXECUTING -> TransferStatus.EXECUTING;
            case PARTLY_APPLIED -> TransferStatus.PARTLY_APPLIED;
        };
    }

    private static VoteType toVoteType(ru.yandex.intranet.d.model.transfers.VoteType type) {
        return switch (type) {
            case CONFIRM -> VoteType.CONFIRM;
            case REJECT -> VoteType.REJECT;
            case ABSTAIN -> VoteType.ABSTAIN;
        };
    }

    private static TransferRequestVoteTypeDto fromVoteType(VoteType voteType) {
        return switch (voteType) {
            case CONFIRM -> TransferRequestVoteTypeDto.CONFIRM;
            case REJECT -> TransferRequestVoteTypeDto.REJECT;
            case ABSTAIN -> TransferRequestVoteTypeDto.ABSTAIN;
            default -> throw new IllegalArgumentException("Unsupported vote type: " + voteType);
        };
    }

    private static TransferRequestTypeDto fromRequestType(TransferType type) {
        return switch (type) {
            case QUOTA_TRANSFER -> TransferRequestTypeDto.QUOTA_TRANSFER;
            case RESERVE_TRANSFER -> TransferRequestTypeDto.RESERVE_TRANSFER;
            case PROVISION_TRANSFER -> TransferRequestTypeDto.PROVISION_TRANSFER;
            default -> throw new IllegalArgumentException("Unsupported transfer type: " + type);
        };
    }

    private static TransferRequestSubtypeDto fromRequestSubtype(TransferSubtype subtype) {
        return switch (subtype) {
            case DEFAULT_QUOTA_TRANSFER -> TransferRequestSubtypeDto.DEFAULT_QUOTA_TRANSFER;
            case DEFAULT_RESERVE_TRANSFER -> TransferRequestSubtypeDto.DEFAULT_RESERVE_TRANSFER;
            case DEFAULT_PROVISION_TRANSFER -> TransferRequestSubtypeDto.DEFAULT_PROVISION_TRANSFER;
            case EXCHANGE_PROVISION_TRANSFER -> TransferRequestSubtypeDto.EXCHANGE_PROVISION_TRANSFER;
            case LOAN_PROVISION_TRANSFER -> TransferRequestSubtypeDto.LOAN_PROVISION_TRANSFER;
            case RESERVE_PROVISION_TRANSFER -> TransferRequestSubtypeDto.RESERVE_PROVISION_TRANSFER;
            case UNRECOGNIZED -> throw new IllegalArgumentException("Unsupported transfer subtype: " + subtype);
        };
    }

}
