package ru.yandex.intranet.d.grpc.services.recipe;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.protobuf.Empty;
import com.google.protobuf.util.Timestamps;
import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.server.service.GrpcService;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Profile;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.backend.service.recipe_proto.Account;
import ru.yandex.intranet.d.backend.service.recipe_proto.AccountKey;
import ru.yandex.intranet.d.backend.service.recipe_proto.AccountName;
import ru.yandex.intranet.d.backend.service.recipe_proto.AccountQuota;
import ru.yandex.intranet.d.backend.service.recipe_proto.AccountsSpaceId;
import ru.yandex.intranet.d.backend.service.recipe_proto.Amount;
import ru.yandex.intranet.d.backend.service.recipe_proto.ApiUri;
import ru.yandex.intranet.d.backend.service.recipe_proto.CreateProviderRequest;
import ru.yandex.intranet.d.backend.service.recipe_proto.CreateServiceRequest;
import ru.yandex.intranet.d.backend.service.recipe_proto.CreateUnitsEnsembleRequest;
import ru.yandex.intranet.d.backend.service.recipe_proto.CreateUserRequest;
import ru.yandex.intranet.d.backend.service.recipe_proto.GetAccountsQuotasResponse;
import ru.yandex.intranet.d.backend.service.recipe_proto.GetAccountsResponse;
import ru.yandex.intranet.d.backend.service.recipe_proto.LastUpdate;
import ru.yandex.intranet.d.backend.service.recipe_proto.MeteringKey;
import ru.yandex.intranet.d.backend.service.recipe_proto.OperationId;
import ru.yandex.intranet.d.backend.service.recipe_proto.Provider;
import ru.yandex.intranet.d.backend.service.recipe_proto.PutProviderRequest;
import ru.yandex.intranet.d.backend.service.recipe_proto.RecipeServiceGrpc;
import ru.yandex.intranet.d.backend.service.recipe_proto.Unit;
import ru.yandex.intranet.d.backend.service.recipe_proto.UnitsEnsemble;
import ru.yandex.intranet.d.backend.service.recipe_proto.Version;
import ru.yandex.intranet.d.grpc.Grpc;
import ru.yandex.intranet.d.grpc.services.CommonProtoConverter;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.services.providers.ProvidersService;
import ru.yandex.intranet.d.services.recipe.RecipeService;
import ru.yandex.intranet.d.services.units.UnitsEnsemblesService;
import ru.yandex.intranet.d.util.AggregationAlgorithmHelper;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.FeatureStateHelper;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.providers.ExternalAccountUrlTemplateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderCreateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderPutDto;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.recipe.AccountsDto;
import ru.yandex.intranet.d.web.model.recipe.AccountsQuotasDto;
import ru.yandex.intranet.d.web.model.recipe.CreateServiceDto;
import ru.yandex.intranet.d.web.model.recipe.CreateUserDto;
import ru.yandex.intranet.d.web.model.units.GrammaticalCaseDto;
import ru.yandex.intranet.d.web.model.units.UnitCreateDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsembleCreateDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * GRPC providers service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@GrpcService
@Profile({"test-recipe"})
public class GrpcRecipeServiceImpl extends RecipeServiceGrpc.RecipeServiceImplBase {

    private final MessageSource messages;
    private final RecipeService recipeService;
    private final FolderService folderService;
    private final ProvidersService providersService;
    private final UnitsEnsemblesService unitsEnsemblesService;
    private final CommonProtoConverter converter;

    public GrpcRecipeServiceImpl(@Qualifier("messageSource") MessageSource messages,
                                 RecipeService recipeService,
                                 FolderService folderService,
                                 ProvidersService providersService,
                                 UnitsEnsemblesService unitsEnsemblesService,
                                 CommonProtoConverter converter) {
        this.messages = messages;
        this.recipeService = recipeService;
        this.folderService = folderService;
        this.providersService = providersService;
        this.unitsEnsemblesService = unitsEnsemblesService;
        this.converter = converter;
    }

    @Override
    public void createService(CreateServiceRequest request, StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> recipeService.createService(fromRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(Empty.newBuilder().build()),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createUser(CreateUserRequest request, StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> recipeService.createUser(fromRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(Empty.newBuilder().build()),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createDefaultFolders(Empty request, StreamObserver<Empty> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> createDefaultFolders(currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(Empty.newBuilder().build()),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getAccounts(Empty request, StreamObserver<GetAccountsResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> recipeService.getAccounts(currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void getAccountsQuotas(Empty request, StreamObserver<GetAccountsQuotasResponse> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> recipeService.getAccountsQuotas(currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createProvider(CreateProviderRequest request, StreamObserver<Provider> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> providersService.create(fromRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void putProvider(PutProviderRequest request, StreamObserver<Provider> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> providersService.put(reqParam.getId(), reqParam.getVersion(),
                        fromRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    @Override
    public void createUnitsEnsemble(CreateUnitsEnsembleRequest request,
                                    StreamObserver<UnitsEnsemble> responseObserver) {
        YaUserDetails currentUser = Auth.grpcUser();
        Locale locale = Locales.grpcLocale();
        Grpc.oneToOne(request, responseObserver, req -> req
                .flatMap(reqParam -> unitsEnsemblesService.create(fromRequest(reqParam), currentUser, locale)
                        .flatMap(resp -> resp.match(u -> Mono.just(toResponse(u)),
                                e -> Mono.error(Errors.toGrpcError(e, messages, locale))))), messages);
    }

    private CreateServiceDto fromRequest(CreateServiceRequest request) {
        return new CreateServiceDto(request.getId(), request.getNameEn(), request.getNameRu(), request.getSlug(),
                request.hasParent() ? request.getParent().getParentId() : null);
    }

    private CreateUserDto fromRequest(CreateUserRequest request) {
        Map<Long, Set<Long>> rolesByService = new HashMap<>();
        request.getServiceRolesList()
                .forEach(r -> rolesByService.put(r.getServiceId(), new HashSet<>(r.getRoleIdsList())));
        return new CreateUserDto(request.getFirstNameEn(), request.getFirstNameRu(), request.getLastNameEn(),
                request.getLastNameRu(), request.getLogin(), request.getUid(), request.getWorkEmail(),
                request.getGender(), request.getLangUi(), request.getTimeZone(), rolesByService, request.getAdmin());
    }

    private ProviderCreateDto fromRequest(CreateProviderRequest request) {
        return new ProviderCreateDto(request.getNameEn(),
                request.getNameRu(),
                request.getDescriptionEn(),
                request.getDescriptionRu(),
                request.hasRestApiUri() ? request.getRestApiUri().getUri() : null,
                request.hasGrpcApiUri() ? request.getGrpcApiUri().getUri() : null,
                request.getSourceTvmId(),
                request.getDestinationTvmId(),
                request.getAbcServiceId(),
                request.getReadOnly(),
                request.getMultipleAccountsPerFolder(),
                request.getAccountTransferWithQuota(),
                request.getManaged(),
                request.getKey(),
                request.getAccountsDisplayNameSupported(),
                request.getAccountsKeySupported(),
                request.getAccountsDeleteSupported(),
                request.getAccountsSoftDeleteSupported(),
                request.getAccountsMoveSupported(),
                request.getAccountsRenameSupported(),
                request.getImportAllowed(),
                request.getAccountsSpacesSupported(),
                request.getPerAccountVersionSupported(),
                request.getPerProvisionVersionSupported(),
                request.getPerAccountLastUpdateSupported(),
                request.getPerProvisionLastUpdateSupported(),
                request.getOperationIdDeduplicationSupported(),
                request.getSyncCoolDownDisabled(),
                request.getRetryCoolDownDisabled(),
                request.getAccountsSyncPageSize(),
                request.getSyncEnabled(),
                request.getGrpcTlsOn(),
                request.hasMeteringKey() ? request.getMeteringKey().getKey() : null,
                request.getTrackerComponentId(),
                request.getMoveProvisionSupported(),
                FeatureStateHelper.fromFeatureState(request.getAllocatedSupported()),
                request.hasAggregationSettings()
                        ? AggregationSettingsHelper.fromAggregationSettings(request.getAggregationSettings())
                        : null,
                request.hasAggregationAlgorithm()
                        ? AggregationAlgorithmHelper.fromAggregationAlgorithm(request.getAggregationAlgorithm())
                        : null,
                request.hasUiSettings()
                        ? new ProviderUISettingsDto(request.getUiSettings())
                        : null,
                request.getExternalAccountUrlTemplateList().stream().map(ExternalAccountUrlTemplateDto::new).toList(),
                FeatureStateHelper.fromFeatureState(request.getMultipleReservesAllowed())
        );
    }

    private ProviderPutDto fromRequest(PutProviderRequest request) {
        return new ProviderPutDto(request.getNameEn(),
                request.getNameRu(),
                request.getDescriptionEn(),
                request.getDescriptionRu(),
                request.hasRestApiUri() ? request.getRestApiUri().getUri() : null,
                request.hasGrpcApiUri() ? request.getGrpcApiUri().getUri() : null,
                request.getSourceTvmId(),
                request.getDestinationTvmId(),
                request.getAbcServiceId(),
                request.getReadOnly(),
                request.getMultipleAccountsPerFolder(),
                request.getAccountTransferWithQuota(),
                request.getManaged(),
                request.getAccountsDisplayNameSupported(),
                request.getAccountsKeySupported(),
                request.getAccountsDeleteSupported(),
                request.getAccountsSoftDeleteSupported(),
                request.getAccountsMoveSupported(),
                request.getAccountsRenameSupported(),
                request.getImportAllowed(),
                request.getAccountsSpacesSupported(),
                request.getPerAccountVersionSupported(),
                request.getPerProvisionVersionSupported(),
                request.getPerAccountLastUpdateSupported(),
                request.getPerProvisionLastUpdateSupported(),
                request.getOperationIdDeduplicationSupported(),
                request.getSyncCoolDownDisabled(),
                request.getRetryCoolDownDisabled(),
                request.getAccountsSyncPageSize(),
                request.getSyncEnabled(),
                request.getGrpcTlsOn(),
                request.hasMeteringKey() ? request.getMeteringKey().getKey() : null,
                request.getTrackerComponentId(),
                request.getMoveProvisionSupported(),
                FeatureStateHelper.fromFeatureState(request.getAllocatedSupported()),
                request.hasAggregationSettings()
                        ? AggregationSettingsHelper.fromAggregationSettings(request.getAggregationSettings())
                        : null,
                request.hasAggregationAlgorithm()
                        ? AggregationAlgorithmHelper.fromAggregationAlgorithm(request.getAggregationAlgorithm())
                        : null,
                null,
                request.getExternalAccountUrlTemplateList().stream().map(ExternalAccountUrlTemplateDto::new).toList(),
                FeatureStateHelper.fromFeatureState(request.getMultipleReservesAllowed())
        );
    }

    private UnitsEnsembleCreateDto fromRequest(CreateUnitsEnsembleRequest request) {
        List<UnitCreateDto> units = request.getUnitsList().stream().map(u -> new UnitCreateDto(
                u.getKey(),
                Map.of(GrammaticalCaseDto.NOMINATIVE, u.getShortNameSingularRuNominative(),
                        GrammaticalCaseDto.GENITIVE, u.getShortNameSingularRuGenitive(),
                        GrammaticalCaseDto.DATIVE, u.getShortNameSingularRuDative(),
                        GrammaticalCaseDto.ACCUSATIVE, u.getShortNameSingularRuAccusative(),
                        GrammaticalCaseDto.INSTRUMENTAL, u.getShortNameSingularRuInstrumental(),
                        GrammaticalCaseDto.PREPOSITIONAL, u.getShortNameSingularRuPrepositional()),
                Map.of(GrammaticalCaseDto.NOMINATIVE, u.getShortNamePluralRuNominative(),
                        GrammaticalCaseDto.GENITIVE, u.getShortNamePluralRuGenitive(),
                        GrammaticalCaseDto.DATIVE, u.getShortNamePluralRuDative(),
                        GrammaticalCaseDto.ACCUSATIVE, u.getShortNamePluralRuAccusative(),
                        GrammaticalCaseDto.INSTRUMENTAL, u.getShortNamePluralRuInstrumental(),
                        GrammaticalCaseDto.PREPOSITIONAL, u.getShortNamePluralRuPrepositional()),
                u.getShortNameSingularEn(),
                u.getShortNamePluralEn(),
                Map.of(GrammaticalCaseDto.NOMINATIVE, u.getLongNameSingularRuNominative(),
                        GrammaticalCaseDto.GENITIVE, u.getLongNameSingularRuGenitive(),
                        GrammaticalCaseDto.DATIVE, u.getLongNameSingularRuDative(),
                        GrammaticalCaseDto.ACCUSATIVE, u.getLongNameSingularRuAccusative(),
                        GrammaticalCaseDto.INSTRUMENTAL, u.getLongNameSingularRuInstrumental(),
                        GrammaticalCaseDto.PREPOSITIONAL, u.getLongNameSingularRuPrepositional()),
                Map.of(GrammaticalCaseDto.NOMINATIVE, u.getLongNamePluralRuNominative(),
                        GrammaticalCaseDto.GENITIVE, u.getLongNamePluralRuGenitive(),
                        GrammaticalCaseDto.DATIVE, u.getLongNamePluralRuDative(),
                        GrammaticalCaseDto.ACCUSATIVE, u.getLongNamePluralRuAccusative(),
                        GrammaticalCaseDto.INSTRUMENTAL, u.getLongNamePluralRuInstrumental(),
                        GrammaticalCaseDto.PREPOSITIONAL, u.getLongNamePluralRuPrepositional()),
                u.getLongNameSingularEn(),
                u.getLongNamePluralEn(),
                u.getBase(),
                u.getPower()
        )).collect(Collectors.toList());
        return new UnitsEnsembleCreateDto(request.getNameEn(),
                request.getNameRu(),
                request.getDescriptionEn(),
                request.getDescriptionRu(),
                request.getFractionsAllowed(),
                units,
                request.getKey());
    }

    private Mono<Result<Void>> createDefaultFolders(YaUserDetails currentUser, Locale locale) {
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        return folderService.addDefaultFolders().thenReturn(Result.success(null));
    }

    private GetAccountsResponse toResponse(AccountsDto response) {
        GetAccountsResponse.Builder builder = GetAccountsResponse.newBuilder();
        response.getAccounts().forEach(account -> {
            Account.Builder accountBuilder = Account.newBuilder()
                    .setId(account.getId())
                    .setProviderId(account.getProviderId())
                    .setExternalId(account.getExternalId())
                    .setFolderId(account.getFolderId())
                    .setDeleted(account.isDeleted())
                    .setFreeTier(account.isFreeTier());
            if (account.getExternalKey().isPresent()) {
                accountBuilder.setExternalKey(AccountKey.newBuilder().setKey(account.getExternalKey().get()).build());
            }
            if (account.getDisplayName().isPresent()) {
                accountBuilder.setDisplayName(AccountName.newBuilder().setName(account.getDisplayName().get()).build());
            }
            if (account.getLastAccountUpdate().isPresent()) {
                accountBuilder.setLastAccountUpdate(LastUpdate.newBuilder().setTimestamp(Timestamps
                        .fromMillis(account.getLastAccountUpdate().get().toEpochMilli())).build());
            }
            if (account.getLastReceivedVersion().isPresent()) {
                accountBuilder.setLastReceivedVersion(Version.newBuilder()
                        .setVersion(account.getLastReceivedVersion().get()).build());
            }
            if (account.getLatestSuccessfulAccountOperationId().isPresent()) {
                accountBuilder.setLatestSuccessfulAccountOperationId(OperationId.newBuilder()
                        .setOperationId(account.getLatestSuccessfulAccountOperationId().get()).build());
            }
            if (account.getAccountsSpacesId().isPresent()) {
                accountBuilder.setAccountsSpaceId(AccountsSpaceId.newBuilder()
                        .setId(account.getAccountsSpacesId().get()).build());
            }
            builder.addAccounts(accountBuilder.build());
        });
        return builder.build();
    }

    private GetAccountsQuotasResponse toResponse(AccountsQuotasDto response) {
        GetAccountsQuotasResponse.Builder builder = GetAccountsQuotasResponse.newBuilder();
        response.getAccountsQuotas().forEach(quota -> {
            AccountQuota.Builder quotaBuilder = AccountQuota.newBuilder()
                    .setAccountId(quota.getAccountId())
                    .setResourceId(quota.getResourceId())
                    .setProvided(Amount.newBuilder()
                            .setValue(quota.getProvidedQuota())
                            .setUnitKey(quota.getProvidedQuotaUnitKey())
                            .build())
                    .setAllocated(Amount.newBuilder()
                            .setValue(quota.getAllocatedQuota())
                            .setUnitKey(quota.getAllocatedQuotaUnitKey())
                            .build())
                    .setFolderId(quota.getFolderId())
                    .setProviderId(quota.getProviderId())
                    .setExternalAccountId(quota.getExternalAccountId());
            if (quota.getLastProvisionUpdate().isPresent()) {
                quotaBuilder.setLastProvisionUpdate(LastUpdate.newBuilder().setTimestamp(Timestamps
                        .fromMillis(quota.getLastProvisionUpdate().get().toEpochMilli())).build());
            }
            if (quota.getLastReceivedProvisionVersion().isPresent()) {
                quotaBuilder.setLastReceivedProvisionVersion(Version.newBuilder()
                        .setVersion(quota.getLastReceivedProvisionVersion().get()).build());
            }
            if (quota.getLatestSuccessfulProvisionOperationId().isPresent()) {
                quotaBuilder.setLatestSuccessfulProvisionOperationId(OperationId.newBuilder()
                        .setOperationId(quota.getLatestSuccessfulProvisionOperationId().get()).build());
            }
            if (quota.getAccountsSpaceId().isPresent()) {
                quotaBuilder.setAccountsSpaceId(AccountsSpaceId.newBuilder()
                        .setId(quota.getAccountsSpaceId().get()).build());
            }
            builder.addAccountsQuotas(quotaBuilder.build());
        });
        return builder.build();
    }

    private Provider toResponse(ProviderModel response) {
        Provider.Builder builder = Provider.newBuilder();
        builder
                .setId(response.getId())
                .setVersion(response.getVersion())
                .setNameEn(response.getNameEn())
                .setNameRu(response.getNameRu())
                .setDescriptionEn(response.getDescriptionEn())
                .setDescriptionRu(response.getDescriptionRu())
                .setSourceTvmId(response.getSourceTvmId())
                .setDestinationTvmId(response.getDestinationTvmId())
                .setAbcServiceId(response.getServiceId())
                .setReadOnly(response.isReadOnly())
                .setMultipleAccountsPerFolder(response.isMultipleAccountsPerFolder())
                .setAccountTransferWithQuota(response.isAccountTransferWithQuota())
                .setManaged(response.isManaged())
                .setKey(response.getKey())
                .setAccountsDisplayNameSupported(response.getAccountsSettings().isDisplayNameSupported())
                .setAccountsKeySupported(response.getAccountsSettings().isKeySupported())
                .setAccountsDeleteSupported(response.getAccountsSettings().isDeleteSupported())
                .setAccountsSoftDeleteSupported(response.getAccountsSettings().isSoftDeleteSupported())
                .setAccountsMoveSupported(response.getAccountsSettings().isMoveSupported())
                .setAccountsRenameSupported(response.getAccountsSettings().isRenameSupported())
                .setImportAllowed(response.isImportAllowed())
                .setAccountsSpacesSupported(response.isAccountsSpacesSupported())
                .setPerAccountVersionSupported(response.getAccountsSettings().isPerAccountVersionSupported())
                .setPerProvisionVersionSupported(response.getAccountsSettings().isPerProvisionVersionSupported())
                .setPerAccountLastUpdateSupported(response.getAccountsSettings().isPerAccountLastUpdateSupported())
                .setPerProvisionLastUpdateSupported(response.getAccountsSettings().isPerProvisionLastUpdateSupported())
                .setOperationIdDeduplicationSupported(response.getAccountsSettings()
                        .isOperationIdDeduplicationSupported())
                .setSyncCoolDownDisabled(response.getAccountsSettings().isSyncCoolDownDisabled())
                .setRetryCoolDownDisabled(response.getAccountsSettings().isRetryCoolDownDisabled())
                .setSyncEnabled(response.isSyncEnabled())
                .setGrpcTlsOn(response.isGrpcTlsOn())
                .setTrackerComponentId(response.getTrackerComponentId())
                .setHasDefaultQuotas(response.hasDefaultQuotas())
                .setAllocatedSupported(FeatureStateHelper.toFeatureState(response.isAllocatedSupported()
                        .orElse(null)))
                .setMultipleReservesAllowed(FeatureStateHelper.toFeatureState(response.getAccountsSettings()
                        .getMultipleReservesAllowed().orElse(null)));
        if (response.getAggregationSettings().isPresent()) {
            builder.setAggregationSettings(AggregationSettingsHelper.toAggregationSettings(response
                    .getAggregationSettings().orElseThrow()));
        }
        if (response.getAggregationAlgorithm().isPresent()) {
            builder.setAggregationAlgorithm(AggregationAlgorithmHelper
                    .toAggregationAlgorithm(response.getAggregationAlgorithm().get()));
        }
        if (response.getRestApiUri().isPresent()) {
            builder.setRestApiUri(ApiUri.newBuilder().setUri(response.getRestApiUri().get()).build());
        }
        if (response.getGrpcApiUri().isPresent()) {
            builder.setGrpcApiUri(ApiUri.newBuilder().setUri(response.getGrpcApiUri().get()).build());
        }
        if (response.getBillingMeta().isPresent() && response.getBillingMeta().get().getMeteringKey().isPresent()) {
            builder.setMeteringKey(MeteringKey.newBuilder()
                    .setKey(response.getBillingMeta().get().getMeteringKey().get()).build());
        }
        if (response.getReserveFolderId().isPresent()) {
            builder.setReserveFolderId(response.getReserveFolderId().get());
        }
        response.getAccountsSettings().getExternalAccountUrlTemplates()
                .forEach(template -> builder.addExternalAccountUrlTemplate(converter.toProto(template)));
        return builder.build();
    }

    private UnitsEnsemble toResponse(UnitsEnsembleModel response) {
        UnitsEnsemble.Builder builder = UnitsEnsemble.newBuilder();
        builder
                .setId(response.getId())
                .setVersion(response.getVersion())
                .setNameEn(response.getNameEn())
                .setNameRu(response.getNameRu())
                .setDescriptionEn(response.getDescriptionEn())
                .setDescriptionRu(response.getDescriptionRu())
                .setFractionsAllowed(response.isFractionsAllowed())
                .setKey(response.getKey());
        response.getUnits().forEach(u -> builder.addUnits(Unit.newBuilder()
                .setId(u.getId())
                .setKey(u.getKey())
                .setShortNameSingularRuNominative(u.getShortNameSingularRu().get(GrammaticalCase.NOMINATIVE))
                .setShortNameSingularRuGenitive(u.getShortNameSingularRu().get(GrammaticalCase.GENITIVE))
                .setShortNameSingularRuDative(u.getShortNameSingularRu().get(GrammaticalCase.DATIVE))
                .setShortNameSingularRuAccusative(u.getShortNameSingularRu().get(GrammaticalCase.ACCUSATIVE))
                .setShortNameSingularRuInstrumental(u.getShortNameSingularRu().get(GrammaticalCase.INSTRUMENTAL))
                .setShortNameSingularRuPrepositional(u.getShortNameSingularRu().get(GrammaticalCase.PREPOSITIONAL))
                .setShortNamePluralRuNominative(u.getShortNamePluralRu().get(GrammaticalCase.NOMINATIVE))
                .setShortNamePluralRuGenitive(u.getShortNamePluralRu().get(GrammaticalCase.GENITIVE))
                .setShortNamePluralRuDative(u.getShortNamePluralRu().get(GrammaticalCase.DATIVE))
                .setShortNamePluralRuAccusative(u.getShortNamePluralRu().get(GrammaticalCase.ACCUSATIVE))
                .setShortNamePluralRuInstrumental(u.getShortNamePluralRu().get(GrammaticalCase.INSTRUMENTAL))
                .setShortNamePluralRuPrepositional(u.getShortNamePluralRu().get(GrammaticalCase.PREPOSITIONAL))
                .setShortNameSingularEn(u.getShortNameSingularEn())
                .setShortNamePluralEn(u.getShortNamePluralEn())
                .setLongNameSingularRuNominative(u.getLongNameSingularRu().get(GrammaticalCase.NOMINATIVE))
                .setLongNameSingularRuGenitive(u.getLongNameSingularRu().get(GrammaticalCase.GENITIVE))
                .setLongNameSingularRuDative(u.getLongNameSingularRu().get(GrammaticalCase.DATIVE))
                .setLongNameSingularRuAccusative(u.getLongNameSingularRu().get(GrammaticalCase.ACCUSATIVE))
                .setLongNameSingularRuInstrumental(u.getLongNameSingularRu().get(GrammaticalCase.INSTRUMENTAL))
                .setLongNameSingularRuPrepositional(u.getLongNameSingularRu().get(GrammaticalCase.PREPOSITIONAL))
                .setLongNamePluralRuNominative(u.getLongNamePluralRu().get(GrammaticalCase.NOMINATIVE))
                .setLongNamePluralRuGenitive(u.getLongNamePluralRu().get(GrammaticalCase.GENITIVE))
                .setLongNamePluralRuDative(u.getLongNamePluralRu().get(GrammaticalCase.DATIVE))
                .setLongNamePluralRuAccusative(u.getLongNamePluralRu().get(GrammaticalCase.ACCUSATIVE))
                .setLongNamePluralRuInstrumental(u.getLongNamePluralRu().get(GrammaticalCase.INSTRUMENTAL))
                .setLongNamePluralRuPrepositional(u.getLongNamePluralRu().get(GrammaticalCase.PREPOSITIONAL))
                .setLongNameSingularEn(u.getLongNameSingularEn())
                .setLongNamePluralEn(u.getLongNamePluralEn())
                .setBase(u.getBase())
                .setPower(u.getPower())
                .build()));
        return builder.build();
    }

}
