package ru.yandex.intranet.d.kotlin

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.reactive.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import kotlinx.coroutines.slf4j.MDCContext
import kotlinx.coroutines.withContext
import mu.KLogger
import org.slf4j.MDC
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.services.integration.providers.ProviderError
import ru.yandex.intranet.d.services.integration.providers.Response
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.web.log.MdcContextSupplier
import java.util.*

/**
 * Helpers.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 12-11-2021
 */

/**
 * Creates a cold [mono][Mono] that runs a given [block] in a coroutine and emits its result.
 * Preserve MDC context.
 */
fun <T> mono(
    block: suspend CoroutineScope.() -> T?
): Mono<T> {
    return kotlinx.coroutines.reactor.mono(MDCContext(), block)
}

suspend fun <T> withMDC(mdc: Map<String, String?>, block: suspend CoroutineScope.() -> T): T {
    val copy = MDC.getCopyOfContextMap()
    copy.putAll(mdc)
    return withContext(MDCContext(copy), block)
}

suspend fun <T> withMDC(vararg pairs: Pair<String, String?>, block: suspend CoroutineScope.() -> T): T {
    val copy = MDC.getCopyOfContextMap()
    pairs.forEach { (key, value) ->
        copy[key] = value
    }
    return withContext(MDCContext(copy), block)
}

suspend inline fun <T> Mono<T>.awaitSingleWithMdc(): T = awaitSingle(MDC.getCopyOfContextMap())

suspend inline fun <T> Mono<T>.awaitSingle(mdc: Map<String, String?>): T =
    contextWrite { it.put(MdcContextSupplier.COMMON_CONTEXT_KEY, mdc) }
        .awaitSingle()

suspend inline fun <T> Mono<T>.awaitSingleOrNullWithMdc(): T? = awaitSingleOrNull(MDC.getCopyOfContextMap())

suspend inline fun <T> Mono<T>.awaitSingleOrNull(mdc: Map<String, String?>): T? =
    contextWrite { it.put(MdcContextSupplier.COMMON_CONTEXT_KEY, mdc) }
        .awaitSingleOrNull()

/**
 * The binding keyword allows multiple calls that each return a Result to be chained imperatively.
 * When inside a binding block, the .bind() function is accessible on any Result.
 * Each call to bind will attempt to unwrap the Result and store its value, returning early if any Result is a Failure.
 *
 * Inspired by https://github.com/michaelbull/kotlin-result
 */
inline fun <T> binding(block: ResultBinding<T>.() -> Result<T>): Result<T> {
    val receiver = ResultBinding<T>()
    return try {
        with(receiver) { block() }
    } catch (ex: BindException) {
        Result.failure(receiver.error)
    }
}

internal object BindException : Exception() {
    override fun fillInStackTrace(): Throwable {
        return this
    }
}

class ResultBinding<T> {
    @PublishedApi
    internal lateinit var error: ErrorCollection

    fun <T2> Result<T2>.bind(): T2? {
        if (isSuccess()) {
            return match({ it }, { null })
        } else {
            doOnFailure { this@ResultBinding.error = it }
            throw BindException
        }
    }
}

fun <T> Optional<T>?.emptyOrNull(): Boolean {
    return this == null || this.isEmpty
}

inline fun <T, X : Throwable> Optional<T>?.getOrThrow(exceptionSupplier: () -> X): T {
    return if (this != null && this.isPresent) {
        this.get()
    } else {
        throw exceptionSupplier()
    }
}

fun <T> Optional<T>?.getOrNull() = this?.orElse(null)

fun Boolean?.orFalse() = this ?: false

inline fun <T> meter(
    logger: KLogger,
    label: String,
    statement: () -> T,
): T {
    return elapsed(statement) { millis, success ->
        logger.info { "$label: duration = $millis ms, success = $success" }
    }
}

suspend inline fun <T, R> Result<Response<T>>.map(
    crossinline action: suspend (data: T, requestId: String?) -> Result<Response<R>>
): Result<Response<R>> {
    return this.matchSuspend(
        { resp ->
            resp.matchSuspend(
                { result: T, requestId: String? -> action(result, requestId) },
                { Result.success(Response.failure(it)) },
                { error: ProviderError, requestId: String? -> Result.success(Response.error(error, requestId)) })
        },
        { Result.failure(it) }
    )
}
