package ru.yandex.intranet.d.kotlin

import com.yandex.ydb.table.transaction.TransactionMode
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.datasource.model.YdbSession
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession

/**
 * Kotlin wrapper for YdbTableClient.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 03-11-2021
 */
@Component
class TableClientAdapter(
    val tableClient: YdbTableClient,
) {
    suspend fun <T> inTransactionOne(
        txMode: TransactionMode = TransactionMode.STALE_READ_ONLY,
        retryMode: RetryMode = RetryMode.RETRYABLE,
        block: suspend CoroutineScope.(YdbTxSession) -> T?
    ): T? {
        return when (retryMode) {
            RetryMode.RETRYABLE ->
                tableClient.usingSessionMonoRetryable { session: YdbSession ->
                    mono {
                        block.invoke(this, session.asTxCommitRetryable(txMode))
                    }
                }.awaitSingleOrNull()
            RetryMode.NON_RETRYABLE ->
                tableClient.usingSessionMono { session: YdbSession ->
                    mono {
                        block.invoke(this, session.asTxCommit(txMode))
                    }
                }.awaitSingleOrNull()
        }
    }

    final suspend inline fun <T> inTransactionOneMono(
        txMode: TransactionMode = TransactionMode.STALE_READ_ONLY,
        retryMode: RetryMode = RetryMode.RETRYABLE,
        crossinline block: (YdbTxSession) -> Mono<T>
    ): T? {
        return when (retryMode) {
            RetryMode.RETRYABLE ->
                tableClient.usingSessionMonoRetryable { session ->
                    block(session.asTxCommitRetryable(txMode))
                }.awaitSingleOrNull()
            RetryMode.NON_RETRYABLE ->
                tableClient.usingSessionMono { session ->
                    block(session.asTxCommit(txMode))
                }.awaitSingleOrNull()
        }
    }

    suspend fun <T> inTransactionMany(
        txMode: TransactionMode = TransactionMode.SERIALIZABLE_READ_WRITE,
        retryMode: RetryMode = RetryMode.RETRYABLE,
        block: suspend CoroutineScope.(YdbTxSession) -> T?
    ): T? {
        when (retryMode) {
            RetryMode.RETRYABLE ->
                return tableClient.usingSessionMonoRetryable { session ->
                    session.usingTxMonoRetryable(txMode) { tx ->
                        mono {
                            block.invoke(this, tx)
                        }
                    }
                }.awaitSingleOrNull()
            RetryMode.NON_RETRYABLE ->
                return tableClient.usingSessionMono { session ->
                    session.usingTxMono(txMode) { tx ->
                        mono {
                            block.invoke(this, tx)
                        }
                    }
                }.awaitSingleOrNull()
        }
    }
}
