package ru.yandex.intranet.d.kotlin.dao

import kotlinx.coroutines.reactor.awaitSingleOrNull
import ru.yandex.intranet.d.dao.AbstractDao
import ru.yandex.intranet.d.dao.Tenants.DEFAULT_TENANT_ID
import ru.yandex.intranet.d.datasource.model.WithTxId
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.TableClientAdapter
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.WithTenant

/**
 * AbstractDaoAdapter.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 21-01-2022
 */
abstract class AbstractDaoAdapter<ModelClass, IdentityClass>(
    private val abstractDao: AbstractDao<ModelClass, IdentityClass>,
    private val tableClient: TableClientAdapter,
) {

    open suspend fun getById(
        session: YdbTxSession, id: IdentityClass, tenantId: TenantId = DEFAULT_TENANT_ID
    ): ModelClass? {
        return abstractDao.getById(session, id, tenantId).awaitSingleOrNull()?.orElse(null)
    }

    open suspend fun getByIdStartTx(
        session: YdbTxSession, id: IdentityClass,
        tenantId: TenantId = DEFAULT_TENANT_ID
    ): WithTxId<ModelClass?> {
        return abstractDao.getByIdStartTx(session, id, tenantId).awaitSingleOrNull()!!.map { it.orElse(null) }
    }

    open suspend fun getByIds(
        session: YdbTxSession,
        ids: List<IdentityClass>,
        tenantId: TenantId = DEFAULT_TENANT_ID
    ): List<ModelClass> {
        return abstractDao.getByIds(session, ids, tenantId).awaitSingleOrNull()!!
    }

    open suspend fun getByIds(session: YdbTxSession, ids: List<WithTenant<IdentityClass>>): List<ModelClass> {
        return abstractDao.getByIds(session, ids).awaitSingleOrNull()!!
    }

    open suspend fun getByIdsStartTx(
        session: YdbTxSession,
        ids: List<IdentityClass>,
        tenantId: TenantId = DEFAULT_TENANT_ID
    ): WithTxId<List<ModelClass>> {
        return abstractDao.getByIdsStartTx(session, ids, tenantId).awaitSingleOrNull()!!
    }

    open suspend fun upsertOneRetryable(session: YdbTxSession, model: ModelClass): ModelClass {
        return abstractDao.upsertOneRetryable(session, model).awaitSingleOrNull()!!
    }

    open suspend fun upsertOneTxRetryable(session: YdbTxSession, model: ModelClass): WithTxId<ModelClass> {
        return abstractDao.upsertOneTxRetryable(session, model).awaitSingleOrNull()!!
    }

    open suspend fun upsertAllRetryable(session: YdbTxSession, models: List<ModelClass>) {
        abstractDao.upsertAllRetryable(session, models).awaitSingleOrNull()
    }

    suspend fun getById(id: IdentityClass, tenantId: TenantId = DEFAULT_TENANT_ID): ModelClass? {
        return tableClient.inTransactionOneMono { tx ->
            abstractDao.getById(tx, id, tenantId)
        }?.orElse(null)
    }

    open suspend fun getByIds(
        ids: List<IdentityClass>,
        tenantId: TenantId = DEFAULT_TENANT_ID
    ): List<ModelClass> {
        return tableClient.inTransactionOneMono { tx ->
            abstractDao.getByIds(tx, ids, tenantId)
        }?: listOf()
    }

}
