package ru.yandex.intranet.d.loaders.providers;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.loaders.CacheKey;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.providers.ProviderId;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;

/**
 * Account spaces loader.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 21.12.2020
 */
@Component
public class AccountSpacesLoader {
    private final ByIdLoader<ProviderId, ProviderSpaces> byIdLoader;
    private final AccountsSpacesDao accountsSpacesDao;

    public AccountSpacesLoader(
            YdbTableClient ydbTableClient,
            AccountsSpacesDao accountsSpacesDao
    ) {
        this.accountsSpacesDao = accountsSpacesDao;
        this.byIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "Account spaces by provider id", 300,
                this::findByIds,
                this::findById,
                ProviderSpaces::getProviderId, ProviderSpaces::getTenantId
        );
    }

    public Mono<Optional<List<AccountSpaceModel>>> getAccountSpaces(
            TenantId tenantId, String providerId, Set<ResourceSegmentSettingsModel> segments
    ) {
        return  getAccountSpacesFromCache(tenantId, providerId, segments)
                .flatMap(accountSpaceModelsOptional -> {
                    if (accountSpaceModelsOptional.isEmpty() || accountSpaceModelsOptional.get().isEmpty()) {
                        return asyncRefresh(new ProviderId(providerId), tenantId)
                                .then(getAccountSpacesFromCache(tenantId, providerId, segments));
                    }

                    return Mono.just(accountSpaceModelsOptional);
                });
    }

    public Mono<List<AccountSpaceModel>> getAllByProviderId(TenantId tenantId, String providerId) {
        return byIdLoader.getByIdImmediate(new ProviderId(providerId), tenantId)
                .map(o -> o.map(ProviderSpaces::getAccountSpaces).orElse(List.of()));
    }

    private Mono<Optional<List<AccountSpaceModel>>>
    getAccountSpacesFromCache(TenantId tenantId, String providerId, Set<ResourceSegmentSettingsModel> segments) {
        return byIdLoader.getByIdImmediate(new ProviderId(providerId), tenantId)
                .map(o -> o.map(spaces -> spaces.getAccountSpaces().stream().filter(accountSpaceModel ->
                        segments.containsAll(accountSpaceModel.getSegments())
                ).collect(Collectors.toList())));
    }

    public Mono<Optional<List<AccountSpaceModel>>> getAccountSpaces(
            YdbTxSession session, TenantId tenantId, String providerId, Set<ResourceSegmentSettingsModel> segments
    ) {
        return  getAccountSpacesFromCache(session, tenantId, providerId, segments)
                .flatMap(accountSpaceModelsOptional -> {
                    if (accountSpaceModelsOptional.isEmpty()) {
                        return asyncRefresh(new ProviderId(providerId), tenantId, session)
                                .then(getAccountSpacesFromCache(session, tenantId, providerId, segments));
                    }

                    return Mono.just(accountSpaceModelsOptional);
                });
    }

    private Mono<Optional<List<AccountSpaceModel>>>
    getAccountSpacesFromCache(YdbTxSession session, TenantId tenantId, String providerId,
                              Set<ResourceSegmentSettingsModel> segments) {
        return byIdLoader.getById(session, new ProviderId(providerId), tenantId)
                .map(o -> o.map(spaces -> spaces.getAccountSpaces().stream().filter(accountSpaceModel ->
                        segments.containsAll(accountSpaceModel.getSegments())
                ).collect(Collectors.toList())));
    }

    public Mono<Optional<List<AccountSpaceModel>>> getAccountSpaces(
            YdbTxSession session,
            TenantId tenantId, String providerId, String id
    ) {
        return byIdLoader.getById(session, new ProviderId(providerId), tenantId)
                .map(o -> o.map(spaces -> spaces.getAccountSpaces().stream().filter(accountSpaceModel ->
                        accountSpaceModel.getId().equals(id)
                ).collect(Collectors.toList())));
    }

    public Mono<Optional<AccountSpaceModel>> getAccountSpacesImmediate(
            TenantId tenantId, String providerId, String id
    ) {
        return byIdLoader.getByIdImmediate(new ProviderId(providerId), tenantId)
                .map(o -> o.map(spaces -> spaces.getAccountSpaces().stream()
                        .filter(accountSpaceModel -> accountSpaceModel.getId().equals(id))
                        .findFirst()
                ).orElse(null));
    }

    public Mono<List<AccountSpaceModel>> getAccountSpaces(
            YdbTxSession session,
            Set<WithTenant<String>> providerIds
    ) {
        List<Tuple2<ProviderId, TenantId>> ids = providerIds.stream()
                .map(pid -> Tuples.of(new ProviderId(pid.getIdentity()), pid.getTenantId()))
                .collect(Collectors.toList());

        return byIdLoader.getByIds(session, ids)
                .map(o -> o.stream()
                        .flatMap(ps -> ps.getAccountSpaces().stream())
                        .collect(Collectors.toList()));
    }

    public void refresh(ProviderId providerId, TenantId tenantId) {
        byIdLoader.refresh(providerId, tenantId);
    }

    public Mono<List<CacheKey<ProviderId>>>
    asyncRefresh(ProviderId providerId, TenantId tenantId) {
        return byIdLoader.asyncRefresh(providerId, tenantId);
    }

    public void refresh(ProviderId providerId, TenantId tenantId, YdbTxSession session) {
        byIdLoader.refresh(providerId, tenantId, session);
    }

    public Mono<List<CacheKey<ProviderId>>>
    asyncRefresh(ProviderId providerId, TenantId tenantId, YdbTxSession session) {
        return byIdLoader.asyncRefresh(providerId, tenantId, session);
    }

    private Mono<Optional<ProviderSpaces>> findById(YdbTxSession ydbTxSession, Tuple2<ProviderId, TenantId> id) {
        ProviderId providerId = id.getT1();
        TenantId tenantId = id.getT2();
        return accountsSpacesDao.getAllByProvider(ydbTxSession, tenantId, providerId.getValue())
                .map(WithTxId::get)
                .map(list -> list.isEmpty() ?
                        Optional.empty() :
                        Optional.of(new ProviderSpaces(providerId, tenantId, list))
                );
    }

    private Mono<List<ProviderSpaces>> findByIds(
            YdbTxSession ydbTxSession,
            List<Tuple2<ProviderId, TenantId>> tuple2s
    ) {
        List<WithTenant<ProviderId>> providerIds =
                tuple2s.stream().map(t -> new WithTenant<>(t.getT2(), t.getT1())).collect(Collectors.toList());
        return accountsSpacesDao
                .getAllByProviderIds(ydbTxSession, providerIds)
                .map(list -> list.get().stream()
                        .collect(Collectors.groupingBy(a ->
                                new WithTenant<>(a.getTenantId(), a.getProviderId())
                        )).entrySet().stream()
                        .map(e -> new ProviderSpaces(
                                new ProviderId(e.getKey().getIdentity()),
                                e.getKey().getTenantId(),
                                e.getValue()
                        ))
                        .collect(Collectors.toList())
                );
    }

    @Scheduled(fixedDelayString = "${caches.resourceSegmentationsCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourceSegmentationsCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
    }

    private static class ProviderSpaces {
        private final ProviderId providerId;
        private final TenantId tenantId;
        private final List<AccountSpaceModel> accountSpaces;

        private ProviderSpaces(ProviderId providerId, TenantId tenantId, List<AccountSpaceModel> accountSpaces) {
            this.providerId = providerId;
            this.tenantId = tenantId;
            this.accountSpaces = accountSpaces;
        }

        public ProviderId getProviderId() {
            return providerId;
        }

        public TenantId getTenantId() {
            return tenantId;
        }

        public List<AccountSpaceModel> getAccountSpaces() {
            return accountSpaces;
        }
    }
}
