package ru.yandex.intranet.d.loaders.providers;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.ProviderModel;

/**
 * Providers loader.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ProvidersLoader {

    private final ByIdLoader<String, ProviderModel> byIdLoader;
    private final ByIdLoader<Long, Tuple3<TenantId, Long, List<ProviderModel>>> bySourceTvmIdLoader;
    private final ProvidersDao providersDao;

    public ProvidersLoader(ProvidersDao providersDao, YdbTableClient ydbTableClient) {
        this.providersDao = providersDao;
        this.byIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "providers by id", 300,
                providersDao::getByIds,
                (ts, id) -> providersDao.getById(ts, id.getT1(), id.getT2()),
                ProviderModel::getId, ProviderModel::getTenantId);
        this.bySourceTvmIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "providers by source TVM id", 300,
                this::getBySourceTvmIds,
                this::getBySourceTvmId,
                Tuple2::getT2, Tuple2::getT1);
    }

    public Mono<Optional<Tuple3<TenantId, Long, List<ProviderModel>>>> getBySourceTvmId(
            YdbTxSession ts, Tuple2<Long, TenantId> id
    ) {
        return providersDao.getBySourceTvmId(ts, id.getT1(), id.getT2())
                .map(this::collectByTvmId)
                .map(list -> list.isEmpty() ? Optional.empty() : Optional.of(list.get(0)));
    }

    private Mono<List<Tuple3<TenantId, Long, List<ProviderModel>>>> getBySourceTvmIds(
            YdbTxSession session,
            List<Tuple2<Long, TenantId>> sourceTvmIds
    ) {
        return providersDao.getBySourceTvmIds(session, sourceTvmIds).map(this::collectByTvmId);
    }

    private List<Tuple3<TenantId, Long, List<ProviderModel>>> collectByTvmId(List<ProviderModel> providers) {
        return providers.stream().collect(Collectors.groupingBy(o -> Tuples.of(o.getTenantId(), o.getSourceTvmId())))
                .entrySet().stream().map(e -> Tuples.of(e.getKey().getT1(), e.getKey().getT2(), e.getValue()))
                .collect(Collectors.toList());
    }

    public Mono<Optional<ProviderModel>> getProviderById(YdbTxSession session, String id, TenantId tenantId) {
        return byIdLoader.getById(session, id, tenantId);
    }

    public Mono<Optional<ProviderModel>> getProviderByIdImmediate(String id, TenantId tenantId) {
        return byIdLoader.getByIdImmediate(id, tenantId);
    }

    public Mono<List<ProviderModel>> getProvidersByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIds(session, ids);
    }

    public Mono<List<ProviderModel>> getProvidersByIdsImmediate(List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIdsImmediate(ids);
    }

    public Mono<List<ProviderModel>> getProviderBySourceTvmId(YdbTxSession session, long sourceTvmId,
                                                                  TenantId tenantId) {
        return bySourceTvmIdLoader.getById(session, sourceTvmId, tenantId)
                .map(o -> o.map(Tuple3::getT3).orElse(List.of()));
    }

    public Mono<List<ProviderModel>> getProviderBySourceTvmIdImmediate(long sourceTvmId, TenantId tenantId) {
        return bySourceTvmIdLoader.getByIdImmediate(sourceTvmId, tenantId)
                .map(o -> o.map(Tuple3::getT3).orElse(List.of()));
    }

    public void update(ProviderModel provider) {
        byIdLoader.update(provider);
        bySourceTvmIdLoader.update(
                Tuples.of(provider.getTenantId(), provider.getSourceTvmId(), List.of(provider)),
                (oldVal, newVal) -> Tuples.of(oldVal.getT1(), oldVal.getT2(),
                        Stream.concat(
                                oldVal.getT3().stream().filter(p -> !p.getId().equals(provider.getId())),
                                Stream.of(provider)
                        ).collect(Collectors.toList())
                )
        );
    }

    @Scheduled(fixedDelayString = "${caches.providersCacheRefreshDelayMs}",
            initialDelayString = "${caches.providersCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
        bySourceTvmIdLoader.refresh();
    }

}
