package ru.yandex.intranet.d.loaders.providers;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.sync.ProvidersSyncStatusDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.metrics.ProvidersSyncMetrics;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel.SyncStatuses;

import static ru.yandex.intranet.d.dao.Tenants.DEFAULT_TENANT_ID;

/**
 * ProvidersSyncStatusHolder.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 23-04-2021
 */
@Component
public class ProvidersSyncStatusHolder {
    private final YdbTableClient tableClient;
    private final ProvidersSyncStatusDao providersSyncStatusDao;
    private final ProvidersLoader providersLoader;
    private final ProvidersSyncMetrics providersSyncMetrics;
    private final ConcurrentMap<String, ProvidersSyncStatusModel> statusByProviderKey = new ConcurrentHashMap<>();

    public ProvidersSyncStatusHolder(
            YdbTableClient tableClient,
            ProvidersSyncStatusDao providersSyncStatusDao,
            ProvidersLoader providersLoader,
            ProvidersSyncMetrics providersSyncMetrics
    ) {
        this.tableClient = tableClient;
        this.providersSyncStatusDao = providersSyncStatusDao;
        this.providersLoader = providersLoader;
        this.providersSyncMetrics = providersSyncMetrics;
    }

    public ProvidersSyncStatusModel getStatus(String providerKey) {
        return statusByProviderKey.get(providerKey);
    }

    @Scheduled(fixedDelayString = "${caches.providersSyncStatusRefreshDelayMs}",
            initialDelayString = "${caches.providersSyncStatusInitialDelayMs}")
    public void refresh() {
        List<ProvidersSyncStatusModel> statuses = tableClient.usingSessionMonoRetryable(session ->
                providersSyncStatusDao.getAll(session).collect(Collectors.toList())
        ).block();
        if (statuses == null || statuses.isEmpty()) {
            return;
        }
        Map<String, String> providersKeyById = getProvidersKeyById(statuses);
        for (ProvidersSyncStatusModel syncStatus : statuses) {
            String providersKey = providersKeyById.get(syncStatus.getProviderId());
            statusByProviderKey.put(providersKey, syncStatus);

            providersSyncMetrics.updateLastSyncErrorStatus(
                    providersKey, SyncStatuses.DONE_OK != syncStatus.getLastSyncStatus()
            );
            if (syncStatus.getNewSyncStart() != null) {
                providersSyncMetrics.updateLastSyncStartTimestamp(providersKey, syncStatus.getNewSyncStart());
            }
            if (syncStatus.getLastSuccessfulSyncFinish() != null) {
                providersSyncMetrics.updateLastSuccessfulSyncTimestamp(
                        providersKey, syncStatus.getLastSuccessfulSyncFinish()
                );
            }
            if (syncStatus.getLastSyncStats() != null) {
                if (syncStatus.getLastSyncStats().getAccountsCount() != null) {
                    providersSyncMetrics.updateAccountsCount(
                            providersKey, syncStatus.getLastSyncStats().getAccountsCount()
                    );
                }
                if (syncStatus.getLastSyncStats().getQuotasCount() != null) {
                    providersSyncMetrics.updateQuotasCount(
                            providersKey, syncStatus.getLastSyncStats().getQuotasCount()
                    );
                }
                if (syncStatus.getLastSyncStats().getSyncDuration() != null) {
                    providersSyncMetrics.updateSyncDuration(
                            providersKey, syncStatus.getLastSyncStats().getSyncDuration()
                    );
                }
            }
        }
    }

    private Map<String, String> getProvidersKeyById(List<ProvidersSyncStatusModel> statuses) {
        List<String> providerIds = statuses.stream().map(ProvidersSyncStatusModel::getProviderId)
                .collect(Collectors.toList());
        List<ProviderModel> providers = providersLoader.getProvidersByIdsImmediate(
                getProviderIdsWithTenant(providerIds)
        ).block();
        if (providers == null || providers.size() < providerIds.size()) {
            var wrongProviderIds = new HashSet<>(providerIds);
            if (providers != null) {
                providers.forEach(p -> wrongProviderIds.remove(p.getId()));
            }
            throw new IllegalStateException("Providers not found: " + wrongProviderIds);
        }
        return providers.stream().collect(Collectors.toMap(ProviderModel::getId, ProviderModel::getKey));
    }

    private List<Tuple2<String, TenantId>> getProviderIdsWithTenant(List<String> providerIds) {
        return providerIds.stream().map(
                providerId -> Tuples.of(providerId, DEFAULT_TENANT_ID)
        ).collect(Collectors.toList());
    }
}
