package ru.yandex.intranet.d.loaders.resources;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoaderByKey;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoaderByKey;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.ResourceBaseIdentity;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceComplexKey;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.imports.SegmentKey;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

/**
 * Loader for lookup resources by complex key.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 09-08-2021
 */
@Component
public class ResourcesByKeysLoader {
    private static final Logger LOG = LoggerFactory.getLogger(ResourcesByKeysLoader.class);

    private final ResourcesDao resourcesDao;
    private final ResourceSegmentationsLoaderByKey resourceSegmentationsLoaderByKey;
    private final ResourceSegmentsLoaderByKey resourceSegmentsLoaderByKey;
    private final ResourceTypesLoader resourceTypesLoader;
    private final ByIdLoader<ResourceBaseIdentity, ResourceEnsemble> resourceEnsembleLoader;
    private final MessageSource messages;

    public ResourcesByKeysLoader(
            ResourcesDao resourcesDao,
            ResourceSegmentationsLoaderByKey resourceSegmentationsLoaderByKey,
            ResourceSegmentsLoaderByKey resourceSegmentsLoaderByKey,
            ResourceTypesLoader resourceTypesLoader,
            YdbTableClient ydbTableClient,
            @Qualifier("messageSource") MessageSource messages
    ) {
        this.resourcesDao = resourcesDao;
        this.resourceSegmentationsLoaderByKey = resourceSegmentationsLoaderByKey;
        this.resourceSegmentsLoaderByKey = resourceSegmentsLoaderByKey;
        this.resourceTypesLoader = resourceTypesLoader;
        this.resourceEnsembleLoader = new ByIdLoader<>(5000, 1000,
                Duration.of(1, ChronoUnit.MINUTES), Duration.of(1, ChronoUnit.MINUTES),
                ydbTableClient, "resources by keys", 300,
                this::getResourceEnsembles,
                this::getResourceEnsemble,
                model -> model.identity,
                model -> model.identity.getTenantId()
        );
        this.messages = messages;
    }

    public Mono<Map<ResourceComplexKey, Result<ResourceModel>>> getResources(
            YdbTxSession ts,
            TenantId tenantId,
            String providerId,
            List<ResourceComplexKey> resourcesKeys,
            Locale locale
    ) {
        return
            resolveSegmentationsByKey(ts, tenantId, providerId, resourcesKeys).flatMap(segmentationsByKey ->
            resolveSegmentsByKey(ts, tenantId, resourcesKeys, segmentationsByKey).flatMap(segmentsByKey ->
            resolveResourceTypesByKey(ts, tenantId, providerId, resourcesKeys).flatMap(resourceTypesByKey ->
            resolveResourceEnsembles(ts, tenantId, providerId, resourceTypesByKey).map(resourcesByResourceTypeId ->
            resolveResourcesByKey(
                    resourcesKeys,
                    segmentationsByKey,
                    segmentsByKey,
                    resourceTypesByKey,
                    resourcesByResourceTypeId,
                    locale
            )))));
    }

    private Map<ResourceComplexKey, Result<ResourceModel>> resolveResourcesByKey(
            List<ResourceComplexKey> resourcesKeys,
            Map<String, ResourceSegmentationModel> segmentationsByKey,
            Map<SegmentKey, ResourceSegmentModel> segmentsByKey,
            Map<String, ResourceTypeModel> resourceTypesByKey,
            Map<String, ResourceEnsemble> resourceEnsemblesByResourceTypeId,
            Locale locale
    ) {
        return resourcesKeys.stream().collect(toMap(identity(), resourceComplexKey -> {
            String resourceTypeId = resourceTypesByKey.get(resourceComplexKey.getResourceTypeKey()).getId();
            if (resourceTypeId == null) {
                LOG.error("Resource type not found. " + resourceComplexKey.getResourceTypeKey());
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(messages.getMessage(
                                "errors.resource.type.not.found", null, locale)))
                        .addDetail("resourceTypeKey", resourceComplexKey.getResourceTypeKey())
                        .build());
            }
            var resourceEnsemble = resourceEnsemblesByResourceTypeId.get(resourceTypeId);
            Set<ResourceSegmentSettingsModel> segmentIds = new HashSet<>();
            for (var entry : resourceComplexKey.getSegmentKeyBySegmentationKey().entrySet()) {
                String segmentationKey = entry.getKey();
                String segmentKey = entry.getValue();
                String segmentationId = segmentationsByKey.get(segmentationKey).getId();
                if (segmentationId == null) {
                    LOG.error("Segmentation not found. " + segmentationKey);
                    return Result.failure(ErrorCollection.builder()
                            .addError(TypedError.invalid(messages.getMessage(
                                    "errors.resource.segmentation.not.found", null, locale)))
                            .addDetail("segmentationKey", segmentationKey)
                            .build());
                }
                String segmentId = segmentsByKey.get(new SegmentKey(segmentationKey, segmentKey)).getId();
                if (segmentId == null) {
                    LOG.error("Segment not found. " + segmentationKey + ": " + segmentKey);
                    return Result.failure(ErrorCollection.builder()
                            .addError(TypedError.invalid(messages.getMessage(
                                    "errors.resource.segment.not.found", null, locale)))
                            .addDetail("segmentationKey", segmentationKey)
                            .addDetail("segmentKey", segmentKey)
                            .build());
                }
                segmentIds.add(new ResourceSegmentSettingsModel(segmentationId, segmentId));
            }
            ResourceModel resource = resourceEnsemble.resourcesBySegments.get(segmentIds);
            if (resource == null) {
                LOG.error("Resource not found. " + resourceComplexKey);
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(messages.getMessage("errors.resource.not.found", null, locale)))
                        .addDetail("segmentIds", segmentIds)
                        .addDetail("resourceEnsemble", resourceEnsemble)
                        .build());
            }
            return Result.success(resource);
        }));
    }

    private Mono<Map<String, ResourceEnsemble>> resolveResourceEnsembles(
            YdbTxSession ts, TenantId tenantId, String providerId,
            Map<String, ResourceTypeModel> resourceTypesByKey
    ) {
        var resourceTypesIds = resourceTypesByKey.values().stream()
                .map(ResourceTypeModel::getId)
                .map(resourceTypeId -> new ResourceBaseIdentity(tenantId, providerId, resourceTypeId))
                .map(o -> Tuples.of(o, tenantId))
                .collect(Collectors.toList());
        return resourceEnsembleLoader.getByIds(ts, resourceTypesIds)
                .map(resourceEnsembles -> resourceEnsembles.stream().collect(toMap(
                        resourceEnsemble -> resourceEnsemble.identity.getResourceTypeId(),
                        identity()
                )));
    }

    private Mono<Map<String, ResourceTypeModel>> resolveResourceTypesByKey(
            YdbTxSession ts, TenantId tenantId, String providerId, List<ResourceComplexKey> resourcesKeys
    ) {
        var resourceTypeKeys = resourcesKeys.stream()
                .map(ResourceComplexKey::getResourceTypeKey)
                .map(resourceTypeKey -> Tuples.of(Tuples.of(providerId, resourceTypeKey), tenantId))
                .collect(Collectors.toList());
        return resourceTypesLoader.getResourceTypesByProviderAndKeys(ts, resourceTypeKeys)
                .map(resourceTypes -> resourceTypes.stream().collect(toMap(ResourceTypeModel::getKey, identity())));
    }

    private Mono<Map<SegmentKey, ResourceSegmentModel>> resolveSegmentsByKey(
            YdbTxSession ts,
            TenantId tenantId,
            List<ResourceComplexKey> resourcesKeys,
            Map<String, ResourceSegmentationModel> segmentationsByKey
    ) {
        List<Tuple2<ResourceSegmentModel.SegmentationAndKey, TenantId>> segmentsKeys = resourcesKeys.stream()
                .flatMap(k -> k.getSegmentKeyBySegmentationKey().entrySet().stream())
                .distinct()
                .map(entry -> {
                    String segmentationKey = entry.getKey();
                    String segmentKey = entry.getValue();
                    return Optional.ofNullable(segmentationsByKey.get(segmentationKey)).map(segmentation ->
                            Tuples.of(
                                    new ResourceSegmentModel.SegmentationAndKey(segmentation.getId(), segmentKey),
                                    tenantId
                            ));
                })
                .filter(Optional::isPresent).map(Optional::get)
                .collect(Collectors.toList());
        Map<String, ResourceSegmentationModel> segmentationsById =
                segmentationsByKey.values().stream().collect(toMap(ResourceSegmentationModel::getId, identity()));
        return resourceSegmentsLoaderByKey.getResourceSegmentsByKeys(ts, segmentsKeys)
                .map(segments -> segments.stream().collect(toMap(
                        segment -> new SegmentKey(
                                segmentationsById.get(segment.getSegmentationId()).getKey(),
                                segment.getKey()
                        ),
                        identity()
                )));
    }

    private Mono<Map<String, ResourceSegmentationModel>> resolveSegmentationsByKey(
            YdbTxSession ts, TenantId tenantId, String providerId, List<ResourceComplexKey> keys
    ) {
        var segmentationKeys = keys.stream()
                .flatMap(k -> k.getSegmentKeyBySegmentationKey().keySet().stream())
                .distinct()
                .map(segmentationKey -> Tuples.of(
                        new ResourceSegmentationModel.ProviderKey(providerId, segmentationKey), tenantId
                )).collect(Collectors.toList());
        return resourceSegmentationsLoaderByKey.getResourceSegmentationByKeys(ts, segmentationKeys)
                .map(segmentations -> segmentations.stream()
                        .collect(toMap(ResourceSegmentationModel::getKey, identity())));
    }

    private Mono<Optional<ResourceEnsemble>> getResourceEnsemble(
            YdbTxSession ts, Tuple2<ResourceBaseIdentity, TenantId> id
    ) {
        return getResourceEnsembles(ts, List.of(id))
                .map(list -> list.isEmpty() ? Optional.empty() : Optional.of(list.get(0)));
    }

    private Mono<List<ResourceEnsemble>> getResourceEnsembles(
            YdbTxSession ts, List<Tuple2<ResourceBaseIdentity, TenantId>> ids
    ) {
        return resourcesDao.getAllByBaseIdentities(ts,
                ids.stream().map(Tuple2::getT1).collect(Collectors.toList()),
                false
        ).map(resources -> resources.stream()
                .collect(Collectors.groupingBy(ResourceBaseIdentity::new)).entrySet().stream()
                .map(entry -> new ResourceEnsemble(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList())
        );
    }

    private static class ResourceEnsemble {
        private final ResourceBaseIdentity identity;
        private final Map<Set<ResourceSegmentSettingsModel>, ResourceModel> resourcesBySegments;

        private ResourceEnsemble(ResourceBaseIdentity identity, List<ResourceModel> resources) {
            this.identity = identity;
            this.resourcesBySegments = resources.stream()
                    .collect(toMap(ResourceModel::getSegments, identity()));
        }
    }

    public void update(ResourceModel resource) {
        resourceEnsembleLoader.refresh(new ResourceBaseIdentity(resource), resource.getTenantId());
    }

    @Scheduled(fixedDelayString = "${caches.resourcesCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourcesCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        resourceEnsembleLoader.refresh();
    }
}
