package ru.yandex.intranet.d.loaders.resources;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.ResourceModel;

/**
 * Resources loader.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourcesLoader {
    private final ByIdLoader<String, ResourceModel> byIdLoader;

    public ResourcesLoader(ResourcesDao resourcesDao, YdbTableClient ydbTableClient) {
        this.byIdLoader = new ByIdLoader<>(5000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "resources", 300,
                resourcesDao::getAllByIds,
                (ts, id) -> resourcesDao.getById(ts, id.getT1(), id.getT2()),
                ResourceModel::getId, ResourceModel::getTenantId);
    }

    public Mono<Optional<ResourceModel>> getResourceById(YdbTxSession session, String id, TenantId tenantId) {
        return byIdLoader.getById(session, id, tenantId);
    }

    public Mono<List<ResourceModel>> getResourcesByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIds(session, ids);
    }

    public Mono<Optional<ResourceModel>> getResourceByIdImmediate(String id, TenantId tenantId) {
        return byIdLoader.getByIdImmediate(id, tenantId);
    }

    public Mono<List<ResourceModel>> getResourcesByIdsImmediate(List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIdsImmediate(ids);
    }

    public void update(ResourceModel resource) {
        byIdLoader.update(resource);
    }

    @Scheduled(fixedDelayString = "${caches.resourcesCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourcesCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
    }
}
