package ru.yandex.intranet.d.loaders.resources.segmentations;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;

/**
 * Resource segmentations loader.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceSegmentationsLoader {

    private final ByIdLoader<String, ResourceSegmentationModel> byIdLoader;

    public ResourceSegmentationsLoader(ResourceSegmentationsDao resourceSegmentationsDao,
                                       YdbTableClient ydbTableClient) {
        this.byIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "resource segmentations", 300,
                resourceSegmentationsDao::getByIds,
                (ts, id) -> resourceSegmentationsDao.getById(ts, id.getT1(), id.getT2()),
                ResourceSegmentationModel::getId, ResourceSegmentationModel::getTenantId);
    }

    public Mono<Optional<ResourceSegmentationModel>> getResourceSegmentationById(YdbTxSession session, String id,
                                                                                 TenantId tenantId) {
        return byIdLoader.getById(session, id, tenantId);
    }

    public Mono<Optional<ResourceSegmentationModel>> getResourceSegmentationByIdImmediate(String id,
                                                                                          TenantId tenantId) {
        return byIdLoader.getByIdImmediate(id, tenantId);
    }

    public Mono<List<ResourceSegmentationModel>> getResourceSegmentationsByIds(
            YdbTxSession session,
            List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIds(session, ids);
    }

    public Mono<List<ResourceSegmentationModel>> getResourceSegmentationsByIdsImmediate(
            List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIdsImmediate(ids);
    }

    public void update(ResourceSegmentationModel resourceSegmentation) {
        byIdLoader.update(resourceSegmentation);
    }

    @Scheduled(fixedDelayString = "${caches.resourceSegmentationsCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourceSegmentationsCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
    }

}
