package ru.yandex.intranet.d.loaders.resources.segmentations;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel.ProviderKey;

/**
 * Resource segmentations loader by provider and key.
 */
@Component
public class ResourceSegmentationsLoaderByKey {

    private final ByIdLoader<ProviderKey, ResourceSegmentationModel> byIdLoader;

    public ResourceSegmentationsLoaderByKey(ResourceSegmentationsDao resourceSegmentationsDao,
                                            YdbTableClient ydbTableClient) {
        this.byIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "resource segmentations by key", 300,
                resourceSegmentationsDao::getAllByProvidersAndKeys,
                (ts, id) -> resourceSegmentationsDao.getAllByProvidersAndKeys(ts, List.of(id))
                        .map(l -> l.stream().findFirst()),
                ResourceSegmentationModel::getProviderKey,
                ResourceSegmentationModel::getTenantId
        );
    }

    public Mono<Optional<ResourceSegmentationModel>> getResourceSegmentationByKey(
            YdbTxSession session, TenantId tenantId, ProviderKey key
    ) {
        return byIdLoader.getById(session, key, tenantId);
    }

    public Mono<Optional<ResourceSegmentationModel>> getResourceSegmentationByKeyImmediate(
            TenantId tenantId, ProviderKey key
    ) {
        return byIdLoader.getByIdImmediate(key, tenantId);
    }

    public Mono<List<ResourceSegmentationModel>> getResourceSegmentationByKeys(
            YdbTxSession session,
            List<Tuple2<ProviderKey, TenantId>> ids
    ) {
        return byIdLoader.getByIds(session, ids);
    }

    public void update(ResourceSegmentationModel resourceSegmentation) {
        byIdLoader.update(resourceSegmentation);
    }

    @Scheduled(fixedDelayString = "${caches.resourceSegmentationsCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourceSegmentationsCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
    }
}
