package ru.yandex.intranet.d.loaders.resources.segments;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;

/**
 * Resource segments loader.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceSegmentsLoader {

    private final ByIdLoader<String, ResourceSegmentModel> byIdLoader;

    public ResourceSegmentsLoader(ResourceSegmentsDao resourceSegmentsDao,
                                  YdbTableClient ydbTableClient) {
        this.byIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "resource segments", 300,
                resourceSegmentsDao::getByIds,
                (ts, id) -> resourceSegmentsDao.getById(ts, id.getT1(), id.getT2()),
                ResourceSegmentModel::getId, ResourceSegmentModel::getTenantId);
    }

    public Mono<Optional<ResourceSegmentModel>> getResourceSegmentById(YdbTxSession session, String id,
                                                                       TenantId tenantId) {
        return byIdLoader.getById(session, id, tenantId);
    }

    public Mono<List<ResourceSegmentModel>> getResourceSegmentsByIds(YdbTxSession session,
                                                                     List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIds(session, ids);
    }

    public Mono<List<ResourceSegmentModel>> getResourceSegmentsByIdsImmediate(List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIdsImmediate(ids);
    }

    public void update(ResourceSegmentModel resourceSegment) {
        byIdLoader.update(resourceSegment);
    }

    @Scheduled(fixedDelayString = "${caches.resourceSegmentsCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourceSegmentsCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
    }

}
