package ru.yandex.intranet.d.loaders.resources.segments;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel.SegmentationAndKey;

/**
 * Resource segments loader by provider and key.
 */
@Component
public class ResourceSegmentsLoaderByKey {

    private final ByIdLoader<SegmentationAndKey, ResourceSegmentModel> byIdLoader;

    public ResourceSegmentsLoaderByKey(ResourceSegmentsDao resourceSegmentsDao,
                                       YdbTableClient ydbTableClient) {
        this.byIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "resource segments by key", 300,
                resourceSegmentsDao::getAllBySegmentationAndKey,
                (ts, id) -> resourceSegmentsDao
                        .getAllBySegmentationAndKey(ts, List.of(id)).map(l -> l.stream().findFirst()),
                ResourceSegmentModel::getSegmentationAndKey,
                ResourceSegmentModel::getTenantId
        );
    }

    public Mono<Optional<ResourceSegmentModel>> getResourceSegmentByKey(
            YdbTxSession session, TenantId tenantId, String segmentationId, String key) {
        return byIdLoader.getById(session, new SegmentationAndKey(segmentationId, key), tenantId);
    }

    public Mono<List<ResourceSegmentModel>> getResourceSegmentsByKeys(
            YdbTxSession session,
            List<Tuple2<SegmentationAndKey, TenantId>> keys
    ) {
        return byIdLoader.getByIds(session, keys);
    }

    public void update(ResourceSegmentModel resourceSegment) {
        byIdLoader.update(resourceSegment);
    }

    @Scheduled(fixedDelayString = "${caches.resourceSegmentsCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourceSegmentsCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
    }
}
