package ru.yandex.intranet.d.loaders.resources.types;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;

import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.ByIdLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;

/**
 * Resource types loader.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceTypesLoader {

    private final ByIdLoader<String, ResourceTypeModel> byIdLoader;
    private final ByIdLoader<Tuple2<String, String>, ResourceTypeModel> byKeyLoader;

    public ResourceTypesLoader(ResourceTypesDao resourceTypesDao, YdbTableClient ydbTableClient) {
        this.byIdLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "resource types", 300,
                resourceTypesDao::getByIds,
                (ts, id) -> resourceTypesDao.getById(ts, id.getT1(), id.getT2()),
                ResourceTypeModel::getId, ResourceTypeModel::getTenantId);
        this.byKeyLoader = new ByIdLoader<>(1000, 1000,
                Duration.of(30, ChronoUnit.MINUTES), Duration.of(30, ChronoUnit.MINUTES),
                ydbTableClient, "resource types by key", 300,
                resourceTypesDao::getByProviderAndKeys,
                (ts, id) -> resourceTypesDao.getByProviderAndKey(ts, id.getT1().getT1(), id.getT1().getT2(),
                        id.getT2()),
                resourceType -> Tuples.of(resourceType.getProviderId(), resourceType.getKey()),
                ResourceTypeModel::getTenantId);
    }

    public Mono<Optional<ResourceTypeModel>> getResourceTypeById(YdbTxSession session, String id,
                                                                 TenantId tenantId) {
        return byIdLoader.getById(session, id, tenantId);
    }

    public Mono<Optional<ResourceTypeModel>> getResourceTypeByIdImmediate(String id, TenantId tenantId) {
        return byIdLoader.getByIdImmediate(id, tenantId);
    }

    public Mono<List<ResourceTypeModel>> getResourceTypesByIds(YdbTxSession session,
                                                               List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIds(session, ids);
    }

    public Mono<List<ResourceTypeModel>> getResourceTypesByIdsImmediate(List<Tuple2<String, TenantId>> ids) {
        return byIdLoader.getByIdsImmediate(ids);
    }

    public void update(ResourceTypeModel resourceType) {
        byIdLoader.update(resourceType);
    }

    @Scheduled(fixedDelayString = "${caches.resourceTypesCacheRefreshDelayMs}",
            initialDelayString = "${caches.resourceTypesCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        byIdLoader.refresh();
        byKeyLoader.refresh();
    }

    public Mono<List<ResourceTypeModel>>
    getResourceTypesByProviderAndKeys(YdbTxSession session, List<Tuple2<Tuple2<String, String>, TenantId>> ids) {
        return byKeyLoader.getByIds(session, ids);
    }
}
