package ru.yandex.intranet.d.loaders.services;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import it.unimi.dsi.fastutil.longs.LongSet;
import org.springframework.stereotype.Component;
import reactor.cache.CacheMono;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Signal;

import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.CacheKey;
import ru.yandex.intranet.d.model.TenantId;

/**
 * Service loader.
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 2-12-2020
 */
@Component
public class ServiceLoader {
    private final ServicesDao servicesDao;
    private final Cache<CacheKey<Long>, LongSet> servicesParentsByServiceId;
    private final YdbTableClient ydbTableClient;

    public ServiceLoader(ServicesDao servicesDao, YdbTableClient ydbTableClient) {
        this.servicesDao = servicesDao;
        this.ydbTableClient = ydbTableClient;
        this.servicesParentsByServiceId = CacheBuilder.newBuilder()
                .maximumSize(1000)
                .expireAfterWrite(10, TimeUnit.MINUTES)
                .build();
    }

    public Mono<LongSet> getAllParentsImmediate(long serviceId, TenantId tenantId) {
        return CacheMono.lookup(this::getParentsFromCacheByServiceId, new CacheKey<>(serviceId, tenantId))
                .onCacheMissResume(() -> loadParentsByServiceIdImmediate(serviceId, tenantId))
                .andWriteWith(this::putParentsByServiceIdToCache);
    }

    private Mono<Signal<? extends LongSet>> getParentsFromCacheByServiceId(CacheKey<Long> key) {
        LongSet longSet = servicesParentsByServiceId.getIfPresent(key);
        if (longSet != null) {
            return Mono.just(Signal.next(longSet));
        }
        return Mono.empty();
    }

    private Mono<LongSet> loadParentsByServiceIdImmediate(long serviceId, TenantId tenantId) {
        return ydbTableClient.usingSessionMonoRetryable(session ->
                servicesDao.getAllParents(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        serviceId, tenantId));
    }

    private Mono<Void> putParentsByServiceIdToCache(CacheKey<Long> key,
                                                   Signal<? extends LongSet> value) {
        return Mono.fromRunnable(() -> {
            if (!value.hasValue()) {
                return;
            }
            LongSet longSet = value.get();
            if (longSet != null) {
                servicesParentsByServiceId.put(key, longSet);
            }
        });
    }


    public Mono<Map<Long, LongSet>> getAllParentsByIds(YdbTxSession session, Set<Long> serviceIds, TenantId tenantId) {
        if (serviceIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<CacheKey<Long>> keys = serviceIds.stream().map(id -> new CacheKey<>(id, tenantId))
                .collect(Collectors.toList());

        return CacheMono.lookup(this::getParentsFromCacheByServiceIds, keys)
                .onCacheMissResume(() -> loadParentsByServiceIds(session, keys, tenantId))
                .andWriteWith((ks, values) -> putParentsByServiceIdsToCache(ks, values, tenantId));
    }

    private Mono<Void> putParentsByServiceIdsToCache(List<CacheKey<Long>> keys,
                                                     Signal<? extends Map<Long, LongSet>> values,
                                                     TenantId tenantId) {
        return Mono.fromRunnable(() -> {
            if (!values.hasValue()) {
                return;
            }
            Map<Long, LongSet> valuesToPut = values.get();
            if (valuesToPut == null) {
                return;
            }
            Map<CacheKey<Long>, LongSet> valuesByKey = valuesToPut.entrySet().stream()
                    .collect(Collectors.toMap(e -> new CacheKey<>(e.getKey(), tenantId),
                            Map.Entry::getValue, (l, r) -> l));

            Map<CacheKey<Long>, LongSet> presentValues = servicesParentsByServiceId.getAllPresent(keys);
            Set<CacheKey<Long>> existingKeys = keys.stream()
                    .filter(valuesByKey::containsKey).collect(Collectors.toSet());
            Set<CacheKey<Long>> keysToAdd = Sets.difference(existingKeys, presentValues.keySet());
            keysToAdd.forEach(k -> servicesParentsByServiceId.put(k, valuesByKey.get(k)));
        });
    }

    private Mono<Map<Long, LongSet>> loadParentsByServiceIds(YdbTxSession session,
                                                             List<CacheKey<Long>> keys,
                                                             TenantId tenantId) {
        ImmutableMap<CacheKey<Long>, LongSet> presentValues = servicesParentsByServiceId.getAllPresent(keys);

        List<Long> serviceIds = keys.stream()
                .filter(k -> !presentValues.containsKey(k))
                .map(CacheKey::getId)
                .collect(Collectors.toList());

        return servicesDao
                .getAllParentsForServices(session, serviceIds, tenantId)
                .map(loaded -> {
                    if (presentValues.isEmpty()) {
                        return loaded;
                    }
                    Map<Long, LongSet> all = new HashMap<>();
                    for (Map.Entry<CacheKey<Long>, LongSet> cacheKeyLongSetEntry : presentValues.entrySet()) {
                        all.put(cacheKeyLongSetEntry.getKey().getId(), cacheKeyLongSetEntry.getValue());
                    }
                    all.putAll(loaded);
                    return all;
                });
    }

    private Mono<Signal<? extends Map<Long, LongSet>>> getParentsFromCacheByServiceIds(
            List<CacheKey<Long>> keys) {

        ImmutableMap<CacheKey<Long>, LongSet> presentKeys = servicesParentsByServiceId.getAllPresent(keys);

        if (keys.stream().allMatch(presentKeys::containsKey)) {
            Map<Long, LongSet> result = new HashMap<>();
            for (Map.Entry<CacheKey<Long>, LongSet> cacheKeyLongSetEntry : presentKeys.entrySet()) {
                result.put(cacheKeyLongSetEntry.getKey().getId(), cacheKeyLongSetEntry.getValue());
            }
            return Mono.just(Signal.next(result));
        }
        return Mono.empty();
    }
}
