package ru.yandex.intranet.d.loaders.users;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import reactor.cache.CacheMono;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Signal;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.CacheKey;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.users.UserModel;

/**
 * Users loader.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class UsersLoader {

    private static final Logger LOG = LoggerFactory.getLogger(UsersLoader.class);

    private final UsersDao usersDao;
    private final Cache<CacheKey<String>, UserModel> usersByPassportUid;
    private final Cache<CacheKey<String>, String> missingUsersByPassportUid;
    private final YdbTableClient ydbTableClient;

    public UsersLoader(UsersDao usersDao, YdbTableClient ydbTableClient) {
        this.usersDao = usersDao;
        this.ydbTableClient = ydbTableClient;
        this.usersByPassportUid = CacheBuilder.newBuilder()
                .maximumSize(10000)
                .expireAfterWrite(30, TimeUnit.MINUTES)
                .build();
        this.missingUsersByPassportUid = CacheBuilder.newBuilder()
                .maximumSize(1000)
                .expireAfterWrite(30, TimeUnit.MINUTES)
                .build();
    }

    public Mono<Optional<UserModel>> getUserByPassportUid(YdbTxSession session, String passportUid,
                                                          TenantId tenantId) {
        return CacheMono.lookup(this::getUserFromCacheByPassportUid, new CacheKey<>(passportUid, tenantId))
                .onCacheMissResume(() -> loadUserByPassportUid(session, passportUid, tenantId))
                .andWriteWith(this::putUserByPassportUidToCache);
    }

    public Mono<Optional<UserModel>> getUserByPassportUidImmediate(String passportUid, TenantId tenantId) {
        return CacheMono.lookup(this::getUserFromCacheByPassportUid, new CacheKey<>(passportUid, tenantId))
                .onCacheMissResume(() -> loadUserByPassportUidImmediate(passportUid, tenantId))
                .andWriteWith(this::putUserByPassportUidToCache);
    }

    @Scheduled(fixedDelayString = "${caches.usersCacheRefreshDelayMs}",
            initialDelayString = "${caches.usersCacheRefreshInitialDelayMs}")
    public void refreshCache() {
        Set<CacheKey<String>> cachedKeysByPassportUid = new HashSet<>(usersByPassportUid
                .asMap().keySet());
        Set<CacheKey<String>> cachedMissingKeysByPassportUid = new HashSet<>(missingUsersByPassportUid
                .asMap().keySet());
        if (cachedKeysByPassportUid.isEmpty() && cachedMissingKeysByPassportUid.isEmpty()) {
            return;
        }
        List<List<Tuple2<String, TenantId>>> usersToLoadByPassportUid = Lists.partition(Sets
                .union(cachedKeysByPassportUid, cachedMissingKeysByPassportUid)
                .stream().map(k -> Tuples.of(k.getId(), k.getTenantId()))
                .collect(Collectors.toList()), 300);
        ydbTableClient.usingSessionMonoRetryable(session ->
                doRefreshByPassportUid(session, usersToLoadByPassportUid)
        ).doOnError(e -> LOG.error("Failed to refresh providers cache", e))
                .onErrorResume(e -> Mono.empty()).block();
    }

    public void updateUser(UserModel userModel) {
        Objects.requireNonNull(userModel, "User must be provided.");
        putToCacheByPassportUid(Collections.singletonList(userModel));
        invalidateMissingUser(userModel);
    }

    private Mono<Void> doRefreshByPassportUid(YdbSession txSession,
                                              List<List<Tuple2<String, TenantId>>> usersToLoad) {
        return Flux.fromIterable(usersToLoad)
                .concatMap(page -> doRefreshByPassportUidPage(txSession, page))
                .flatMapIterable(l -> l)
                .collectList()
                .doOnSuccess(missingUsersByPassportUid::invalidateAll).then();
    }

    private Mono<List<CacheKey<String>>> doRefreshByPassportUidPage(YdbSession session,
                                                                    List<Tuple2<String, TenantId>> page) {
        return usersDao.getByPassportUids(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), page)
                .doOnSuccess(this::putToCacheByPassportUid)
                .map(l -> l.stream().filter(u -> u.getPassportUid().isPresent())
                        .map(u -> new CacheKey<>(u.getPassportUid().get(), u.getTenantId()))
                        .collect(Collectors.toList()));
    }

    private void putToCacheByPassportUid(List<UserModel> users) {
        users.stream().filter(u -> u.getPassportUid().isPresent()).forEach(user -> {
            CacheKey<String> key = new CacheKey<>(user.getPassportUid().get(), user.getTenantId());
            usersByPassportUid.put(key, user);
        });
    }

    private void invalidateMissingUser(UserModel user) {
        if (user.getPassportUid().isPresent()) {
            CacheKey<String> key = new CacheKey<>(user.getPassportUid().get(), user.getTenantId());
            missingUsersByPassportUid.invalidate(key);
        }
    }

    private Mono<Signal<? extends Optional<UserModel>>> getUserFromCacheByPassportUid(CacheKey<String> key) {
        UserModel user = usersByPassportUid.getIfPresent(key);
        if (user != null) {
            return Mono.just(Signal.next(Optional.of(user)));
        }
        String missingId = missingUsersByPassportUid.getIfPresent(key);
        if (missingId != null) {
            return Mono.just(Signal.next(Optional.empty()));
        }
        return Mono.empty();
    }

    private Mono<Optional<UserModel>> loadUserByPassportUid(YdbTxSession session, String passportUid,
                                                            TenantId tenantId) {
        return usersDao.getByPassportUid(session, passportUid, tenantId);
    }

    private Mono<Optional<UserModel>> loadUserByPassportUidImmediate(String passportUid, TenantId tenantId) {
        return ydbTableClient.usingSessionMonoRetryable(session ->
                usersDao.getByPassportUid(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        passportUid, tenantId));
    }

    private Mono<Void> putUserByPassportUidToCache(CacheKey<String> key,
                                                   Signal<? extends Optional<UserModel>> value) {
        return Mono.fromRunnable(() -> {
            if (!value.hasValue()) {
                return;
            }
            Optional<UserModel> user = value.get();
            if (user != null && user.isPresent()) {
                usersByPassportUid.put(key, user.get());
            } else {
                missingUsersByPassportUid.put(key, key.getId());
            }
        });
    }

}
