package ru.yandex.intranet.d.metrics;

import java.util.concurrent.ConcurrentHashMap;

import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * HTTP server metrics.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class HttpServerMetrics {

    private static final String RATE = "http.server.requests_rate";
    private static final String DURATION = "http.server.requests_duration_millis";
    private static final String SIZE = "http.server.response_size_bytes";
    private static final String CODE = "http_code";
    private static final String ALL = "all";
    private static final String UNKNOWN = "unknown";

    private final Rate totalRate;
    private final Histogram totalDuration;
    private final Histogram totalSize;
    private final ConcurrentHashMap<String, Rate> perCodeRate = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Histogram> perCodeDuration = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Histogram> perCodeSize = new ConcurrentHashMap<>();

    public HttpServerMetrics() {
        this.totalRate = MetricRegistry.root().rate(RATE, Labels.of(CODE, ALL));
        this.totalDuration = MetricRegistry.root().histogramRate(DURATION, Labels.of(CODE, ALL),
                Histograms.exponential(22, 2.0d, 1.0d));
        this.totalSize = MetricRegistry.root().histogramRate(SIZE, Labels.of(CODE, ALL),
                Histograms.exponential(12, 10.0d, 1.0d));
        perCodeRate.put("500", MetricRegistry.root().rate(RATE, Labels.of(CODE, "500")));
    }

    public void onRequestCompletion(String code, long durationMillis, Long responseSizeBytes) {
        totalRate.inc();
        totalDuration.record(durationMillis);
        if (responseSizeBytes != null) {
            totalSize.record(responseSizeBytes);
        }
        String codeName = code != null ? code : UNKNOWN;
        Rate codeRate = perCodeRate.computeIfAbsent(codeName,
                key -> MetricRegistry.root().rate(RATE, Labels.of(CODE, codeName)));
        codeRate.inc();
        Histogram codeDuration = perCodeDuration.computeIfAbsent(codeName,
                key -> MetricRegistry.root().histogramRate(DURATION, Labels.of(CODE, codeName),
                        Histograms.exponential(22, 2.0d, 1.0d)));
        codeDuration.record(durationMillis);
        if (responseSizeBytes != null) {
            Histogram codeSize = perCodeSize.computeIfAbsent(codeName,
                    key -> MetricRegistry.root().histogramRate(SIZE, Labels.of(CODE, codeName),
                            Histograms.exponential(12, 10.0d, 1.0d)));
            codeSize.record(responseSizeBytes);
        }
    }

}
