package ru.yandex.intranet.d.metrics;

import java.util.Optional;

import javax.annotation.PostConstruct;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.logbroker.agent.client.SessionMetrics;
import ru.yandex.logbroker.agent.log4j2.appender.UnifiedAgentAppender;
import ru.yandex.monlib.metrics.JvmGc;
import ru.yandex.monlib.metrics.JvmMemory;
import ru.yandex.monlib.metrics.JvmRuntime;
import ru.yandex.monlib.metrics.JvmThreads;
import ru.yandex.monlib.metrics.Metric;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * Metrics initializer
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class MetricsInitializer {

    private static final Logger LOG = LoggerFactory.getLogger(MetricsInitializer.class);

    @PostConstruct
    public void postConstruct() {
        MetricRegistry rootRegistry = MetricRegistry.root();
        JvmGc.addMetrics(rootRegistry);
        JvmMemory.addMetrics(rootRegistry);
        JvmRuntime.addMetrics(rootRegistry);
        JvmThreads.addMetrics(rootRegistry);
        try {
            prepareUnifiedAgentMetrics(rootRegistry);
        } catch (Exception e) {
            LOG.warn("Failed to collect unified agent metrics", e);
        }
    }

    private void prepareUnifiedAgentMetrics(MetricRegistry rootRegistry) {
        LoggerContext ctx = (LoggerContext) LogManager.getContext(false);
        Optional<UnifiedAgentAppender> unifiedAgentAppender = ctx.getConfiguration()
                .getAppenders().values().stream()
                .filter(appender -> appender instanceof UnifiedAgentAppender)
                .map(appender -> (UnifiedAgentAppender) appender)
                .findFirst();
        unifiedAgentAppender.ifPresent(appender -> {
            MetricRegistry metricRegistry = appender.getMetricRegistry();
            copyMetric(metricRegistry, rootRegistry, SessionMetrics.ACTIVE_SESSIONS_COUNT,
                    "unified.agent.active.sessions.count");
            MetricRegistry sourceSubRegistry = metricRegistry.subRegistry("session", "default");
            MetricRegistry destinationSubRegistry = rootRegistry.subRegistry("session", "default");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.RECEIVED_MESSAGES,
                    "unified.agent.received.messages");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.RECEIVED_BYTES,
                    "unified.agent.received.bytes");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.ACKNOWLEDGED_MESSAGES,
                    "unified.agent.acknowledged.messages");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.ACKNOWLEDGED_BYTES,
                    "unified.agent.acknowledged.bytes");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.INFLIGHT_MESSAGES,
                    "unified.agent.in.flight.messages");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.INFLIGHT_BYTES,
                    "unified.agent.in.flight.bytes");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.GRPC_WRITE_BATCH_REQUESTS,
                    "unified.agent.grpc.write.batch.requests");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.GRPC_INFLIGHT_MESSAGES,
                    "unified.agent.grpc.in.flight.messages");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.GRPC_INFLIGHT_BYTES,
                    "unified.agent.grpc.in.flight.bytes");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.GRPC_CALLS,
                    "unified.agent.grpc.calls");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.GRPC_CALLS_INITIALIZED,
                    "unified.agent.grpc.calls.initialized");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.DROPPED_MESSAGES,
                    "unified.agent.dropped.messages");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.DROPPED_BYTES,
                    "unified.agent.dropped.bytes");
            copyMetric(sourceSubRegistry, destinationSubRegistry, SessionMetrics.ERRORS_COUNT,
                    "unified.agent.errors.count");
        });
    }

    private void copyMetric(MetricRegistry sourceRegistry, MetricRegistry destinationRegistry,
                            String sourceKey, String destinationKey) {
        Metric metric = sourceRegistry.getMetric(new MetricId(sourceKey, Labels.of()));
        if (metric != null) {
            destinationRegistry.putMetricIfAbsent(new MetricId(destinationKey, Labels.of()), metric);
        }
    }

}
