package ru.yandex.intranet.d.metrics;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * Provider sync metrics.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 23-04-2021
 */
@Component
public class ProvidersSyncMetrics {
    private static final String ACCOUNTS_COUNT_RATE_NAME = "providers_sync.accounts_count";
    private static final String QUOTAS_COUNT_RATE_NAME = "providers_sync.quotas_count";
    private static final String SYNC_DURATION_RATE_NAME = "providers_sync.sync_duration_millis";
    private static final String TIME_SINCE_LAST_SUCCESSFUL_SYNC_RATE_NAME =
            "providers_sync.time_since_last_successful_sync_millis";
    private static final String TIME_SINCE_LAST_SYNC_START_RATE_NAME =
            "providers_sync.time_since_last_sync_start_millis";
    private static final String LAST_SYNC_ERROR_STATUS_GAUGE_NAME = "providers_sync.last_sync_error";
    private static final String PROVIDER_LABEL_KEY = "provider";

    private final ConcurrentHashMap<String, GaugeInt64> accountsCountByProviderKey = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, GaugeInt64> quotasCountByProviderKey = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, GaugeInt64> syncDurationByProviderKey = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, GaugeInt64> timeSinceLastSuccessfulSyncByProviderKey =
            new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, GaugeInt64> timeSinceLastSyncStartByProviderKey = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, GaugeInt64> lastSyncErrorStatusByProviderKey = new ConcurrentHashMap<>();

    public void updateAccountsCount(String providerKey, long accountsCount) {
        GaugeInt64 gauge = accountsCountByProviderKey.computeIfAbsent(providerKey, provider ->
                MetricRegistry.root().gaugeInt64(ACCOUNTS_COUNT_RATE_NAME, Labels.of(PROVIDER_LABEL_KEY, provider))
        );
        gauge.set(accountsCount);
    }

    public void updateQuotasCount(String providerKey, long quotasCount) {
        GaugeInt64 gauge = quotasCountByProviderKey.computeIfAbsent(providerKey, provider ->
                MetricRegistry.root().gaugeInt64(QUOTAS_COUNT_RATE_NAME, Labels.of(PROVIDER_LABEL_KEY, provider))
        );
        gauge.set(quotasCount);
    }

    public void updateSyncDuration(String providerKey, Duration syncDuration) {
        GaugeInt64 gauge = syncDurationByProviderKey.computeIfAbsent(providerKey, provider ->
                MetricRegistry.root().gaugeInt64(SYNC_DURATION_RATE_NAME, Labels.of(PROVIDER_LABEL_KEY, provider))
        );
        gauge.set(syncDuration.toMillis());
    }

    public void updateLastSuccessfulSyncTimestamp(String providerKey, Instant timestamp) {
        long millis = Duration.between(timestamp, Instant.now()).toMillis();
        GaugeInt64 gauge = timeSinceLastSuccessfulSyncByProviderKey.computeIfAbsent(providerKey, provider ->
                MetricRegistry.root().gaugeInt64(
                        TIME_SINCE_LAST_SUCCESSFUL_SYNC_RATE_NAME, Labels.of(PROVIDER_LABEL_KEY, provider)
                )
        );
        gauge.set(millis);
    }

    public void updateLastSyncStartTimestamp(String providerKey, Instant timestamp) {
        long millis = Duration.between(timestamp, Instant.now()).toMillis();
        GaugeInt64 gauge = timeSinceLastSyncStartByProviderKey.computeIfAbsent(providerKey, provider ->
                MetricRegistry.root().gaugeInt64(
                        TIME_SINCE_LAST_SYNC_START_RATE_NAME, Labels.of(PROVIDER_LABEL_KEY, provider)
                )
        );
        gauge.set(millis);
    }

    public void updateLastSyncErrorStatus(String providerKey, boolean isError) {
        GaugeInt64 gauge = lastSyncErrorStatusByProviderKey.computeIfAbsent(providerKey, provider ->
                MetricRegistry.root().gaugeInt64(
                        LAST_SYNC_ERROR_STATUS_GAUGE_NAME, Labels.of(PROVIDER_LABEL_KEY, provider)
                )
        );
        gauge.set(isError ? 1 : 0);
    }

    public GaugeInt64 getAccountsCount(String providerKey) {
        return accountsCountByProviderKey.get(providerKey);
    }
}
