package ru.yandex.intranet.d.metrics;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.yandex.ydb.table.stats.SessionPoolStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.utils.ReadinessYdbTableClientHolder;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * YDB session metrics.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class YdbSessionMetrics {

    private static final Logger LOG = LoggerFactory.getLogger(YdbSessionMetrics.class);

    private static final String POOL_MIN_SIZE = "ydb.pool.min_size";
    private static final String POOL_MAX_SIZE = "ydb.pool.max_size";
    private static final String POOL_IDLE_COUNT = "ydb.pool.idle_count";
    private static final String POOL_DISCONNECTED_COUNT = "ydb.pool.disconnected_count";
    private static final String POOL_ACQUIRED_COUNT = "ydb.pool.acquired_count";
    private static final String POOL_PENDING_ACQUIRE_COUNT = "ydb.pool.pending_acquire_count";
    private static final String READINESS_POOL_MIN_SIZE = "ydb.readiness.pool.min_size";
    private static final String READINESS_POOL_MAX_SIZE = "ydb.readiness.pool.max_size";
    private static final String READINESS_POOL_IDLE_COUNT = "ydb.readiness.pool.idle_count";
    private static final String READINESS_POOL_DISCONNECTED_COUNT = "ydb.readiness.pool.disconnected_count";
    private static final String READINESS_POOL_ACQUIRED_COUNT = "ydb.readiness.pool.acquired_count";
    private static final String READINESS_POOL_PENDING_ACQUIRE_COUNT = "ydb.readiness.pool.pending_acquire_count";

    private final YdbTableClient tableClient;
    private final YdbTableClient readinessTableClient;
    private final ScheduledExecutorService scheduler;

    private volatile ScheduledFuture<?> scheduledFuture;
    private volatile int poolMinSize;
    private volatile int poolMaxSize;
    private volatile int poolIdleCount;
    private volatile int poolDisconnectedCount;
    private volatile int poolAcquiredCount;
    private volatile int poolPendingAcquireCount;
    private volatile int readinessPoolMinSize;
    private volatile int readinessPoolMaxSize;
    private volatile int readinessPoolIdleCount;
    private volatile int readinessPoolDisconnectedCount;
    private volatile int readinessPoolAcquiredCount;
    private volatile int readinessPoolPendingAcquireCount;

    public YdbSessionMetrics(YdbTableClient tableClient,
                             ReadinessYdbTableClientHolder readinessTableClient) {
        this.tableClient = tableClient;
        this.readinessTableClient = readinessTableClient.getTableClient();
        ThreadFactory threadFactory = new ThreadFactoryBuilder()
                .setDaemon(true)
                .setNameFormat("ydb-metrics-scheduler-pool-%d")
                .setUncaughtExceptionHandler((t, e) ->
                        LOG.error("Uncaught exception in YDB metrics thread " + t, e))
                .build();
        ScheduledThreadPoolExecutor scheduledThreadPoolExecutor = new ScheduledThreadPoolExecutor(2,
                threadFactory);
        scheduledThreadPoolExecutor.setRemoveOnCancelPolicy(true);
        this.scheduler = scheduledThreadPoolExecutor;
        MetricRegistry.root().lazyGaugeInt64(POOL_MIN_SIZE, () -> poolMinSize);
        MetricRegistry.root().lazyGaugeInt64(POOL_MAX_SIZE, () -> poolMaxSize);
        MetricRegistry.root().lazyGaugeInt64(POOL_IDLE_COUNT, () -> poolIdleCount);
        MetricRegistry.root().lazyGaugeInt64(POOL_DISCONNECTED_COUNT, () -> poolDisconnectedCount);
        MetricRegistry.root().lazyGaugeInt64(POOL_ACQUIRED_COUNT, () -> poolAcquiredCount);
        MetricRegistry.root().lazyGaugeInt64(POOL_PENDING_ACQUIRE_COUNT, () -> poolPendingAcquireCount);
        MetricRegistry.root().lazyGaugeInt64(READINESS_POOL_MIN_SIZE, () -> readinessPoolMinSize);
        MetricRegistry.root().lazyGaugeInt64(READINESS_POOL_MAX_SIZE, () -> readinessPoolMaxSize);
        MetricRegistry.root().lazyGaugeInt64(READINESS_POOL_IDLE_COUNT, () -> readinessPoolIdleCount);
        MetricRegistry.root().lazyGaugeInt64(READINESS_POOL_DISCONNECTED_COUNT, () -> readinessPoolDisconnectedCount);
        MetricRegistry.root().lazyGaugeInt64(READINESS_POOL_ACQUIRED_COUNT, () -> readinessPoolAcquiredCount);
        MetricRegistry.root().lazyGaugeInt64(READINESS_POOL_PENDING_ACQUIRE_COUNT,
                () -> readinessPoolPendingAcquireCount);
    }

    @PostConstruct
    @SuppressWarnings("FutureReturnValueIgnored")
    public void postConstruct() {
        scheduledFuture = scheduler.scheduleWithFixedDelay(this::updateMetrics, 1, 1, TimeUnit.SECONDS);
    }

    @PreDestroy
    public void preDestroy() {
        if (scheduledFuture != null) {
            scheduledFuture.cancel(true);
            scheduledFuture = null;
        }
        scheduler.shutdown();
        try {
            scheduler.awaitTermination(1, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        scheduler.shutdownNow();
    }

    private void updateMetrics() {
        try {
            SessionPoolStats stats = tableClient.getSessionPoolStats();
            poolMinSize = stats.getMinSize();
            poolMaxSize = stats.getMaxSize();
            poolIdleCount = stats.getIdleCount();
            poolDisconnectedCount = stats.getDisconnectedCount();
            poolAcquiredCount = stats.getAcquiredCount();
            poolPendingAcquireCount = stats.getPendingAcquireCount();
            SessionPoolStats readinessStats = readinessTableClient.getSessionPoolStats();
            readinessPoolMinSize = readinessStats.getMinSize();
            readinessPoolMaxSize = readinessStats.getMaxSize();
            readinessPoolIdleCount = readinessStats.getIdleCount();
            readinessPoolDisconnectedCount = readinessStats.getDisconnectedCount();
            readinessPoolAcquiredCount = readinessStats.getAcquiredCount();
            readinessPoolPendingAcquireCount = readinessStats.getPendingAcquireCount();
        } catch (Exception e) {
            LOG.error("Failed to update YDB session pool metrics");
        }
    }

}
