package ru.yandex.intranet.d.model.accounts;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Account.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 15.10.2020
 */
public final class AccountModel {
    private final TenantId tenantId;
    private final String id;
    private final long version;
    private final String providerId;
    private final String outerAccountIdInProvider;
    private final String outerAccountKeyInProvider;
    private final String folderId;
    private final String displayName;
    private final boolean deleted;
    private final Instant lastAccountUpdate;
    private final Long lastReceivedVersion;
    private final String latestSuccessfulAccountOperationId;
    @Nullable
    private final String accountsSpacesId;
    private final boolean freeTier;
    @Nullable
    private final AccountReserveType reserveType;

    public AccountModel(Builder builder) {
        this.tenantId = builder.tenantId;
        this.id = builder.id;
        this.version = builder.version;
        this.providerId = builder.providerId;
        this.outerAccountIdInProvider = builder.outerAccountIdInProvider;
        this.outerAccountKeyInProvider = builder.outerAccountKeyInProvider;
        this.folderId = builder.folderId;
        this.displayName = builder.displayName;
        this.deleted = builder.deleted;
        this.lastAccountUpdate = builder.lastAccountUpdate;
        this.lastReceivedVersion = builder.lastReceivedVersion;
        this.latestSuccessfulAccountOperationId = builder.latestSuccessfulAccountOperationId;
        this.accountsSpacesId = builder.accountsSpacesId;
        this.freeTier = builder.freeTier;
        this.reserveType = builder.reserveType;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getId() {
        return id;
    }

    public long getVersion() {
        return version;
    }

    public String getProviderId() {
        return providerId;
    }

    public String getOuterAccountIdInProvider() {
        return outerAccountIdInProvider;
    }

    public Optional<String> getOuterAccountKeyInProvider() {
        return Optional.ofNullable(outerAccountKeyInProvider);
    }

    public String getFolderId() {
        return folderId;
    }

    public Optional<String> getDisplayName() {
        return Optional.ofNullable(displayName);
    }

    public boolean isDeleted() {
        return deleted;
    }

    public Instant getLastAccountUpdate() {
        return lastAccountUpdate;
    }

    public Optional<Long> getLastReceivedVersion() {
        return Optional.ofNullable(lastReceivedVersion);
    }

    public Optional<String> getLatestSuccessfulAccountOperationId() {
        return Optional.ofNullable(latestSuccessfulAccountOperationId);
    }

    public Optional<String> getAccountsSpacesId() {
        return Optional.ofNullable(accountsSpacesId);
    }

    public boolean isFreeTier() {
        return freeTier;
    }

    public Optional<AccountReserveType> getReserveType() {
        return Optional.ofNullable(reserveType);
    }

    public ExternalId toExternalId() {
        return new ExternalId(providerId, outerAccountIdInProvider, accountsSpacesId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountModel that = (AccountModel) o;
        return version == that.version &&
                deleted == that.deleted &&
                freeTier == that.freeTier &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(id, that.id) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(outerAccountIdInProvider, that.outerAccountIdInProvider) &&
                Objects.equals(outerAccountKeyInProvider, that.outerAccountKeyInProvider) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(lastAccountUpdate, that.lastAccountUpdate) &&
                Objects.equals(lastReceivedVersion, that.lastReceivedVersion) &&
                Objects.equals(latestSuccessfulAccountOperationId, that.latestSuccessfulAccountOperationId) &&
                Objects.equals(accountsSpacesId, that.accountsSpacesId) &&
                Objects.equals(reserveType, that.reserveType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, id, version, providerId, outerAccountIdInProvider, outerAccountKeyInProvider,
                folderId, displayName, deleted, lastAccountUpdate, lastReceivedVersion,
                latestSuccessfulAccountOperationId, accountsSpacesId, freeTier, reserveType);
    }

    @Override
    public String toString() {
        return "AccountModel{" +
                "tenantId=" + tenantId +
                ", id='" + id + '\'' +
                ", version=" + version +
                ", providerId='" + providerId + '\'' +
                ", outerAccountIdInProvider='" + outerAccountIdInProvider + '\'' +
                ", outerAccountKeyInProvider='" + outerAccountKeyInProvider + '\'' +
                ", folderId='" + folderId + '\'' +
                ", displayName='" + displayName + '\'' +
                ", deleted=" + deleted +
                ", lastAccountUpdate=" + lastAccountUpdate +
                ", lastReceivedVersion=" + lastReceivedVersion +
                ", latestSuccessfulAccountOperationId='" + latestSuccessfulAccountOperationId + '\'' +
                ", accountsSpacesId='" + accountsSpacesId + '\'' +
                ", freeTier=" + freeTier +
                ", reserveType=" + reserveType +
                '}';
    }

    public static final class ExternalId {

        private final String providerId;
        private final String outerAccountIdInProvider;
        @Nullable
        private final String accountsSpacesId;

        public ExternalId(String providerId, String outerAccountIdInProvider, @Nullable String accountsSpacesId) {
            this.providerId = providerId;
            this.outerAccountIdInProvider = outerAccountIdInProvider;
            this.accountsSpacesId = accountsSpacesId;
        }

        public String getProviderId() {
            return providerId;
        }

        public String getOuterAccountIdInProvider() {
            return outerAccountIdInProvider;
        }

        public Optional<String> getAccountsSpacesId() {
            return Optional.ofNullable(accountsSpacesId);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ExternalId that = (ExternalId) o;
            return Objects.equals(providerId, that.providerId) &&
                    Objects.equals(outerAccountIdInProvider, that.outerAccountIdInProvider) &&
                    Objects.equals(accountsSpacesId, that.accountsSpacesId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(providerId, outerAccountIdInProvider, accountsSpacesId);
        }

        @Override
        public String toString() {
            return "ExternalId{" +
                    "providerId='" + providerId + '\'' +
                    ", outerAccountIdInProvider='" + outerAccountIdInProvider + '\'' +
                    ", accountsSpacesId='" + accountsSpacesId + '\'' +
                    '}';
        }

    }

    public static class Builder {
        private TenantId tenantId;
        private String id;
        private long version = 0;
        private String providerId;
        private String outerAccountIdInProvider;
        private String outerAccountKeyInProvider;
        private String folderId;
        private String displayName;
        private boolean deleted = false;
        private Instant lastAccountUpdate;
        private Long lastReceivedVersion;
        private String latestSuccessfulAccountOperationId;
        private String accountsSpacesId;
        private boolean freeTier = false;
        private AccountReserveType reserveType;

        public Builder() {
        }

        public Builder(AccountModel model) {
            this.tenantId = model.tenantId;
            this.id = model.id;
            this.version = model.version;
            this.providerId = model.providerId;
            this.outerAccountIdInProvider = model.outerAccountIdInProvider;
            this.outerAccountKeyInProvider = model.outerAccountKeyInProvider;
            this.folderId = model.folderId;
            this.displayName = model.displayName;
            this.deleted = model.deleted;
            this.lastAccountUpdate = model.lastAccountUpdate;
            this.lastReceivedVersion = model.lastReceivedVersion;
            this.latestSuccessfulAccountOperationId = model.latestSuccessfulAccountOperationId;
            this.accountsSpacesId = model.accountsSpacesId;
            this.freeTier = model.freeTier;
            this.reserveType = model.reserveType;
        }

        public Builder setTenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public Builder setVersion(long version) {
            this.version = version;
            return this;
        }

        public Builder setProviderId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder setOuterAccountIdInProvider(String outerAccountIdInProvider) {
            this.outerAccountIdInProvider = outerAccountIdInProvider;
            return this;
        }

        public Builder setOuterAccountKeyInProvider(String outerAccountKeyInProvider) {
            this.outerAccountKeyInProvider = outerAccountKeyInProvider;
            return this;
        }

        public Builder setFolderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder setDisplayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder setDeleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder setLastAccountUpdate(Instant lastAccountUpdate) {
            this.lastAccountUpdate = lastAccountUpdate;
            return this;
        }

        public Builder setLastReceivedVersion(Long lastReceivedVersion) {
            this.lastReceivedVersion = lastReceivedVersion;
            return this;
        }

        public Builder setLatestSuccessfulAccountOperationId(String latestSuccessfulAccountOperationId) {
            this.latestSuccessfulAccountOperationId = latestSuccessfulAccountOperationId;
            return this;
        }

        public Builder setAccountsSpacesId(String accountsSpacesId) {
            this.accountsSpacesId = accountsSpacesId;
            return this;
        }

        public Builder setFreeTier(boolean freeTier) {
            this.freeTier = freeTier;
            return this;
        }

        public Builder setReserveType(AccountReserveType reserveType) {
            this.reserveType = reserveType;
            return this;
        }

        public AccountModel build() {
            return new AccountModel(this);
        }
    }
}
