package ru.yandex.intranet.d.model.accounts;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Accounts quotas model.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 20.10.2020
 */
@SuppressWarnings("unused")
public final class AccountsQuotasModel {
    private final TenantId tenantId;
    private final Identity identity;
    private final Long providedQuota;
    private final Long allocatedQuota;
    private final String folderId;
    private final String providerId;
    private final Instant lastProvisionUpdate;
    private final Long lastReceivedProvisionVersion;
    private final String latestSuccessfulProvisionOperationId;
    private final long frozenProvidedQuota;

    public AccountsQuotasModel(Builder builder) {
        this.tenantId = builder.tenantId;
        identity = new Identity(
                builder.accountId,
                builder.resourceId
        );
        this.providedQuota = builder.providedQuota;
        this.allocatedQuota = builder.allocatedQuota;
        this.folderId = builder.folderId;
        this.providerId = builder.providerId;
        this.lastProvisionUpdate = builder.lastProvisionUpdate;
        this.lastReceivedProvisionVersion = builder.lastReceivedProvisionVersion;
        this.latestSuccessfulProvisionOperationId = builder.latestSuccessfulProvisionOperationId;
        this.frozenProvidedQuota = builder.frozenProvidedQuota;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getAccountId() {
        return identity.accountId;
    }

    public String getResourceId() {
        return identity.resourceId;
    }

    public Long getProvidedQuota() {
        return providedQuota;
    }

    public Long getAllocatedQuota() {
        return allocatedQuota;
    }

    public String getFolderId() {
        return folderId;
    }

    public String getProviderId() {
        return providerId;
    }

    public Instant getLastProvisionUpdate() {
        return lastProvisionUpdate;
    }

    public Optional<Long> getLastReceivedProvisionVersion() {
        return Optional.ofNullable(lastReceivedProvisionVersion);
    }

    public Optional<String> getLatestSuccessfulProvisionOperationId() {
        return Optional.ofNullable(latestSuccessfulProvisionOperationId);
    }

    public long getFrozenProvidedQuota() {
        return frozenProvidedQuota;
    }

    public Identity getIdentity() {
        return identity;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountsQuotasModel that = (AccountsQuotasModel) o;
        return frozenProvidedQuota == that.frozenProvidedQuota &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(identity, that.identity) &&
                Objects.equals(providedQuota, that.providedQuota) &&
                Objects.equals(allocatedQuota, that.allocatedQuota) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(lastProvisionUpdate, that.lastProvisionUpdate) &&
                Objects.equals(lastReceivedProvisionVersion, that.lastReceivedProvisionVersion) &&
                Objects.equals(latestSuccessfulProvisionOperationId, that.latestSuccessfulProvisionOperationId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, identity, providedQuota, allocatedQuota, folderId, providerId,
                lastProvisionUpdate, lastReceivedProvisionVersion, latestSuccessfulProvisionOperationId,
                frozenProvidedQuota);
    }

    @Override
    public String toString() {
        return "AccountsQuotasModel{" +
                "tenantId=" + tenantId +
                ", identity=" + identity +
                ", providedQuota=" + providedQuota +
                ", allocatedQuota=" + allocatedQuota +
                ", folderId='" + folderId + '\'' +
                ", providerId='" + providerId + '\'' +
                ", lastProvisionUpdate=" + lastProvisionUpdate +
                ", lastReceivedProvisionVersion=" + lastReceivedProvisionVersion +
                ", latestSuccessfulProvisionOperationId='" + latestSuccessfulProvisionOperationId + '\'' +
                ", frozenProvidedQuota=" + frozenProvidedQuota +
                '}';
    }

    public Builder copyBuilder() {
        return new Builder()
                .setTenantId(tenantId)
                .setAccountId(getAccountId())
                .setResourceId(getResourceId())
                .setProvidedQuota(providedQuota)
                .setAllocatedQuota(allocatedQuota)
                .setFrozenProvidedQuota(frozenProvidedQuota)
                .setFolderId(folderId)
                .setProviderId(providerId)
                .setLastProvisionUpdate(lastProvisionUpdate)
                .setLastReceivedProvisionVersion(lastReceivedProvisionVersion)
                .setLatestSuccessfulProvisionOperationId(latestSuccessfulProvisionOperationId);
    }

    public static class Builder {
        private TenantId tenantId;
        private String accountId;
        private String resourceId;
        private Long providedQuota;
        private Long allocatedQuota;
        private String folderId;
        private String providerId;
        private Instant lastProvisionUpdate;
        private Long lastReceivedProvisionVersion;
        private String latestSuccessfulProvisionOperationId;
        private long frozenProvidedQuota = 0L;

        public Builder() {
        }

        public Builder(AccountsQuotasModel model) {
            this.tenantId = model.tenantId;
            this.accountId = model.identity.accountId;
            this.resourceId = model.identity.resourceId;
            this.providedQuota = model.providedQuota;
            this.allocatedQuota = model.allocatedQuota;
            this.folderId = model.folderId;
            this.providerId = model.providerId;
            this.lastProvisionUpdate = model.lastProvisionUpdate;
            this.lastReceivedProvisionVersion = model.lastReceivedProvisionVersion;
            this.latestSuccessfulProvisionOperationId = model.latestSuccessfulProvisionOperationId;
            this.frozenProvidedQuota = model.frozenProvidedQuota;
        }

        public Builder setTenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder setAccountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public Builder setResourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder setProvidedQuota(Long providedQuota) {
            this.providedQuota = providedQuota;
            return this;
        }

        public Builder setAllocatedQuota(Long allocatedQuota) {
            this.allocatedQuota = allocatedQuota;
            return this;
        }

        public Builder setFolderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder setProviderId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder setLastProvisionUpdate(Instant lastProvisionUpdate) {
            this.lastProvisionUpdate = lastProvisionUpdate;
            return this;
        }

        public Builder setLastReceivedProvisionVersion(Long lastReceivedProvisionVersion) {
            this.lastReceivedProvisionVersion = lastReceivedProvisionVersion;
            return this;
        }

        public Builder setLatestSuccessfulProvisionOperationId(String latestSuccessfulProvisionOperationId) {
            this.latestSuccessfulProvisionOperationId = latestSuccessfulProvisionOperationId;
            return this;
        }

        public Builder setFrozenProvidedQuota(long frozenProvidedQuota) {
            this.frozenProvidedQuota = frozenProvidedQuota;
            return this;
        }

        public AccountsQuotasModel build() {
            return new AccountsQuotasModel(this);
        }
    }

    public static final class Identity {

        private final String accountId;
        private final String resourceId;

        public Identity(String accountId, String resourceId) {
            this.accountId = Objects.requireNonNull(accountId);
            this.resourceId = Objects.requireNonNull(resourceId);
        }

        public String getAccountId() {
            return accountId;
        }

        public String getResourceId() {
            return resourceId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Identity identity = (Identity) o;
            return accountId.equals(identity.accountId) &&
                    resourceId.equals(identity.resourceId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(accountId, resourceId);
        }

        @Override
        public String toString() {
            return "Identity{" +
                    "accountId='" + accountId + '\'' +
                    ", resourceId='" + resourceId + '\'' +
                    '}';
        }
    }
}
