package ru.yandex.intranet.d.model.accounts;

import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.springframework.http.HttpStatus;

import ru.yandex.intranet.d.model.TenantId;

/**
 * AccountsQuotasOperationsModel.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 22.10.2020
 */
public final class AccountsQuotasOperationsModel {
    private final TenantId tenantId;
    private final String operationId;
    private final String lastRequestId;
    private final Instant createDateTime;
    private final OperationSource operationSource;
    private final OperationType operationType;
    private final String authorUserId;
    private final String authorUserUid;
    private final String providerId;
    private final String accountsSpaceId;
    private final Instant updateDateTime;
    private final RequestStatus requestStatus;
    private final String errorMessage;
    private final OperationErrorCollections fullErrorMessage;
    private final OperationChangesModel requestedChanges;
    private final OperationOrdersModel orders;
    private final OperationErrorKind errorKind;
    private final List<AccountsQuotasOperationsRequestLogModel> logs;

    public AccountsQuotasOperationsModel(Builder builder) {
        this.tenantId = builder.tenantId;
        this.operationId = builder.operationId;
        this.lastRequestId = builder.lastRequestId;
        this.createDateTime = builder.createDateTime;
        this.operationSource = builder.operationSource;
        this.operationType = builder.operationType;
        this.authorUserId = builder.authorUserId;
        this.authorUserUid = builder.authorUserUid;
        this.providerId = builder.providerId;
        this.accountsSpaceId = builder.accountsSpaceId;
        this.updateDateTime = builder.updateDateTime;
        this.requestStatus = builder.requestStatus;
        this.errorMessage = builder.errorMessage;
        this.fullErrorMessage = builder.fullErrorMessage;
        this.requestedChanges = builder.requestedChanges;
        this.orders = builder.orders;
        this.errorKind = builder.errorKind;
        this.logs = builder.logs;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getOperationId() {
        return operationId;
    }

    public Optional<String> getLastRequestId() {
        return Optional.ofNullable(lastRequestId);
    }

    public Instant getCreateDateTime() {
        return createDateTime;
    }

    public OperationSource getOperationSource() {
        return operationSource;
    }

    public OperationType getOperationType() {
        return operationType;
    }

    public String getAuthorUserId() {
        return authorUserId;
    }

    public Optional<String> getAuthorUserUid() {
        return Optional.ofNullable(authorUserUid);
    }

    public String getProviderId() {
        return providerId;
    }

    public Optional<String> getAccountsSpaceId() {
        return Optional.ofNullable(accountsSpaceId);
    }

    public Optional<Instant> getUpdateDateTime() {
        return Optional.ofNullable(updateDateTime);
    }

    public Optional<RequestStatus> getRequestStatus() {
        return Optional.ofNullable(requestStatus);
    }

    public Optional<String> getErrorMessage() {
        return Optional.ofNullable(errorMessage);
    }

    public Optional<OperationErrorCollections> getFullErrorMessage() {
        return Optional.ofNullable(fullErrorMessage);
    }

    public OperationChangesModel getRequestedChanges() {
        return requestedChanges;
    }

    public OperationOrdersModel getOrders() {
        return orders;
    }

    public Optional<OperationErrorKind> getErrorKind() {
        return Optional.ofNullable(errorKind);
    }

    public List<AccountsQuotasOperationsRequestLogModel> getLogs() {
        return logs;
    }
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountsQuotasOperationsModel that = (AccountsQuotasOperationsModel) o;
        return Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(operationId, that.operationId) &&
                Objects.equals(lastRequestId, that.lastRequestId) &&
                Objects.equals(createDateTime, that.createDateTime) &&
                operationSource == that.operationSource &&
                operationType == that.operationType &&
                Objects.equals(authorUserId, that.authorUserId) &&
                Objects.equals(authorUserUid, that.authorUserUid) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(accountsSpaceId, that.accountsSpaceId) &&
                Objects.equals(updateDateTime, that.updateDateTime) &&
                requestStatus == that.requestStatus &&
                Objects.equals(errorMessage, that.errorMessage) &&
                Objects.equals(fullErrorMessage, that.fullErrorMessage) &&
                Objects.equals(requestedChanges, that.requestedChanges) &&
                Objects.equals(orders, that.orders) &&
                errorKind == that.errorKind;
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, operationId, lastRequestId, createDateTime, operationSource, operationType,
                authorUserId, authorUserUid, providerId, accountsSpaceId, updateDateTime, requestStatus, errorMessage,
                fullErrorMessage, requestedChanges, orders, errorKind);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {

        private TenantId tenantId;
        private String operationId;
        private String lastRequestId;
        private Instant createDateTime;
        private OperationSource operationSource;
        private OperationType operationType;
        private String authorUserId;
        private String authorUserUid;
        private String providerId;
        private String accountsSpaceId;
        private Instant updateDateTime;
        private RequestStatus requestStatus;
        private String errorMessage;
        private OperationErrorCollections fullErrorMessage;
        private OperationChangesModel requestedChanges;
        private OperationOrdersModel orders;
        private OperationErrorKind errorKind;
        private List<AccountsQuotasOperationsRequestLogModel> logs;

        public Builder() {
        }

        public Builder(AccountsQuotasOperationsModel value) {
            this.tenantId = value.tenantId;
            this.operationId = value.operationId;
            this.lastRequestId = value.lastRequestId;
            this.createDateTime = value.createDateTime;
            this.operationSource = value.operationSource;
            this.operationType = value.operationType;
            this.authorUserId = value.authorUserId;
            this.authorUserUid = value.authorUserUid;
            this.providerId = value.providerId;
            this.accountsSpaceId = value.accountsSpaceId;
            this.updateDateTime = value.updateDateTime;
            this.requestStatus = value.requestStatus;
            this.errorMessage = value.errorMessage;
            this.fullErrorMessage = value.fullErrorMessage;
            this.requestedChanges = value.requestedChanges;
            this.orders = value.orders;
            this.errorKind = value.errorKind;
            this.logs = value.logs;
        }

        public Builder setTenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder setOperationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public Builder setLastRequestId(String lastRequestId) {
            this.lastRequestId = lastRequestId;
            return this;
        }

        public Builder setCreateDateTime(Instant createDateTime) {
            this.createDateTime = createDateTime;
            return this;
        }

        public Builder setOperationSource(OperationSource operationSource) {
            this.operationSource = operationSource;
            return this;
        }

        public Builder setOperationType(OperationType operationType) {
            this.operationType = operationType;
            return this;
        }

        public Builder setAuthorUserId(String authorUserId) {
            this.authorUserId = authorUserId;
            return this;
        }

        public Builder setAuthorUserUid(String authorUserUid) {
            this.authorUserUid = authorUserUid;
            return this;
        }

        public Builder setProviderId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder setAccountsSpaceId(String accountsSpaceId) {
            this.accountsSpaceId = accountsSpaceId;
            return this;
        }

        public Builder setUpdateDateTime(Instant updateDateTime) {
            this.updateDateTime = updateDateTime;
            return this;
        }

        public Builder setRequestStatus(RequestStatus requestStatus) {
            this.requestStatus = requestStatus;
            return this;
        }

        public Builder setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public Builder setFullErrorMessage(OperationErrorCollections fullErrorMessage) {
            this.fullErrorMessage = fullErrorMessage;
            return this;
        }

        public Builder setRequestedChanges(OperationChangesModel requestedChanges) {
            this.requestedChanges = requestedChanges;
            return this;
        }

        public Builder setOrders(OperationOrdersModel orders) {
            this.orders = orders;
            return this;
        }

        public Builder setErrorKind(OperationErrorKind errorKind) {
            this.errorKind = errorKind;
            return this;
        }

        public Builder setLogs(List<AccountsQuotasOperationsRequestLogModel> logs) {
            this.logs = logs;
            return this;
        }

        public Optional<OperationType> getOperationType() {
            return Optional.ofNullable(operationType);
        }

        public Optional<RequestStatus> getRequestStatus() {
            return Optional.ofNullable(requestStatus);
        }

        public Optional<OperationChangesModel> getRequestedChanges() {
            return Optional.ofNullable(requestedChanges);
        }

        public Optional<OperationOrdersModel> getOrders() {
            return Optional.ofNullable(orders);
        }

        public AccountsQuotasOperationsModel build() {
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(operationId, "OperationId is required");
            Preconditions.checkNotNull(createDateTime, "CreateDateTime is required");
            Preconditions.checkNotNull(operationSource, "OperationSource is required");
            Preconditions.checkNotNull(operationType, "OperationType is required");
            Preconditions.checkNotNull(authorUserId, "AuthorUserId is required");
            Preconditions.checkNotNull(providerId, "ProviderId is required");
            Preconditions.checkNotNull(requestedChanges, "RequestChanges is required");
            Preconditions.checkNotNull(orders, "Orders is required");
            return new AccountsQuotasOperationsModel(this);
        }
    }

    public enum OperationType {
        UPDATE_PROVISION,
        DELIVER_AND_UPDATE_PROVISION,
        CREATE_ACCOUNT,
        CREATE_ACCOUNT_AND_PROVIDE,
        DELETE_ACCOUNT,
        RENAME_ACCOUNT,
        MOVE_ACCOUNT,
        MOVE_PROVISION,
        PUT_ACCOUNT,
        PROVIDE_RESERVE
    }

    public enum RequestStatus {
        WAITING,
        OK(HttpStatus.OK),
        ERROR(HttpStatus.INTERNAL_SERVER_ERROR);

        private final HttpStatus httpResponseStatus;

        RequestStatus() {
            this(null);
        }

        RequestStatus(HttpStatus httpResponseStatus) {
            this.httpResponseStatus = httpResponseStatus;
        }

        public HttpStatus getHttpResponseStatus() {
            return httpResponseStatus;
        }
    }
}
