package ru.yandex.intranet.d.model.accounts;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.util.result.ErrorCollection;

/**
 * Error collection for operation
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public class OperationErrorCollections {
    private final Map<String, OperationErrorCollection> errorCollectionByLocaleMap;

    @JsonCreator
    public OperationErrorCollections(Map<String, OperationErrorCollection> errorCollectionByLocaleMap) {
        this.errorCollectionByLocaleMap = errorCollectionByLocaleMap;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Map<String, OperationErrorCollection> getErrorCollectionByLocaleMap() {
        return errorCollectionByLocaleMap;
    }

    public OperationErrorCollection getErrorCollection(Locale locale) {
        return errorCollectionByLocaleMap.getOrDefault(createKey(locale),
                errorCollectionByLocaleMap.get(createKey(Locales.ENGLISH)));
    }

    private static String createKey(Locale locale) {
        return locale.toLanguageTag();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OperationErrorCollections that = (OperationErrorCollections) o;
        return Objects.equals(errorCollectionByLocaleMap, that.errorCollectionByLocaleMap);
    }

    @Override
    public int hashCode() {
        return Objects.hash(errorCollectionByLocaleMap);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public static final class Builder {

        private final Map<String, OperationErrorCollection> errorCollectionByLocaleMap = new HashMap<>();

        private Builder() {
        }

        public Builder addErrorCollection(Locale locale, OperationErrorCollection operationErrorCollection) {
            errorCollectionByLocaleMap.put(createKey(locale), operationErrorCollection);
            return this;
        }

        public Builder addErrorCollection(Locale locale, ErrorCollection errorCollection) {
            errorCollectionByLocaleMap.put(createKey(locale), new OperationErrorCollection(errorCollection));
            return this;
        }

        public OperationErrorCollections build() {
            return new OperationErrorCollections(errorCollectionByLocaleMap);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Builder builder = (Builder) o;
            return Objects.equals(errorCollectionByLocaleMap, builder.errorCollectionByLocaleMap);
        }

        @Override
        public int hashCode() {
            return Objects.hash(errorCollectionByLocaleMap);
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }
}
