package ru.yandex.intranet.d.model.accounts;

import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Operation in progress.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class OperationInProgressModel {

    private final TenantId tenantId;
    private final String operationId;
    private final String folderId;
    private final String accountId;
    private final long retryCounter;

    public OperationInProgressModel(
            TenantId tenantId,
            String operationId,
            String folderId,
            String accountId,
            long retryCounter
    ) {
        this.tenantId = tenantId;
        this.operationId = operationId;
        this.folderId = folderId;
        this.accountId = accountId;
        this.retryCounter = retryCounter;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(OperationInProgressModel value) {
        return new Builder(value);
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getOperationId() {
        return operationId;
    }

    public String getFolderId() {
        return folderId;
    }

    public Optional<String> getAccountId() {
        return Optional.ofNullable(accountId);
    }

    public long getRetryCounter() {
        return retryCounter;
    }

    public Key getKey() {
        return new Key(operationId, folderId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OperationInProgressModel that = (OperationInProgressModel) o;
        return retryCounter == that.retryCounter &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(operationId, that.operationId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(accountId, that.accountId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, operationId, folderId, accountId, retryCounter);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public static final class Key {

        private final String operationId;
        private final String folderId;

        public Key(String operationId, String folderId) {
            this.operationId = operationId;
            this.folderId = folderId;
        }

        public String getOperationId() {
            return operationId;
        }

        public String getFolderId() {
            return folderId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return Objects.equals(operationId, key.operationId) &&
                    Objects.equals(folderId, key.folderId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(operationId, folderId);
        }

        @Override
        public String toString() {
            return "Key{" +
                    "operationId='" + operationId + '\'' +
                    ", folderId='" + folderId + '\'' +
                    '}';
        }

    }

    public static final class Builder {

        private TenantId tenantId;
        private String operationId;
        private String folderId;
        private String accountId;
        private long retryCounter = 0;

        private Builder() {
        }

        private Builder(OperationInProgressModel value) {
            this.tenantId = value.tenantId;
            this.operationId = value.operationId;
            this.folderId = value.folderId;
            this.accountId = value.accountId;
            this.retryCounter = value.retryCounter;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public Builder retryCounter(long retryCounter) {
            this.retryCounter = retryCounter;
            return this;
        }

        public OperationInProgressModel build() {
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(operationId, "OperationId is required");
            Preconditions.checkNotNull(folderId, "FolderId is required");
            return new OperationInProgressModel(tenantId, operationId, folderId, accountId, retryCounter);
        }

    }

}
