package ru.yandex.intranet.d.model.accounts;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Operation orders.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public final class OperationOrdersModel {

    private final long submitOrder;
    private final Long closeOrder;
    private final Long restoreOrder;
    private final Long destinationSubmitOrder;
    private final Long destinationCloseOrder;
    private final Long destinationRestoreOrder;

    @JsonCreator
    public OperationOrdersModel(@JsonProperty("submitOrder") long submitOrder,
                                @JsonProperty("closeOrder") Long closeOrder,
                                @JsonProperty("restoreOrder") Long restoreOrder,
                                @JsonProperty("destinationSubmitOrder") Long destinationSubmitOrder,
                                @JsonProperty("destinationCloseOrder") Long destinationCloseOrder,
                                @JsonProperty("destinationRestoreOrder") Long destinationRestoreOrder) {
        this.submitOrder = submitOrder;
        this.closeOrder = closeOrder;
        this.restoreOrder = restoreOrder;
        this.destinationSubmitOrder = destinationSubmitOrder;
        this.destinationCloseOrder = destinationCloseOrder;
        this.destinationRestoreOrder = destinationRestoreOrder;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(OperationOrdersModel value) {
        return new Builder(value);
    }

    public Builder copyBuilder() {
        return new Builder(this);
    }

    public long getSubmitOrder() {
        return submitOrder;
    }

    public Optional<Long> getCloseOrder() {
        return Optional.ofNullable(closeOrder);
    }

    public Optional<Long> getRestoreOrder() {
        return Optional.ofNullable(restoreOrder);
    }

    public Optional<Long> getDestinationSubmitOrder() {
        return Optional.ofNullable(destinationSubmitOrder);
    }

    public Optional<Long> getDestinationCloseOrder() {
        return Optional.ofNullable(destinationCloseOrder);
    }

    public Optional<Long> getDestinationRestoreOrder() {
        return Optional.ofNullable(destinationRestoreOrder);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OperationOrdersModel that = (OperationOrdersModel) o;
        return submitOrder == that.submitOrder &&
                Objects.equals(closeOrder, that.closeOrder) &&
                Objects.equals(restoreOrder, that.restoreOrder) &&
                Objects.equals(destinationSubmitOrder, that.destinationSubmitOrder) &&
                Objects.equals(destinationCloseOrder, that.destinationCloseOrder) &&
                Objects.equals(destinationRestoreOrder, that.destinationRestoreOrder);
    }

    @Override
    public int hashCode() {
        return Objects.hash(submitOrder, closeOrder, restoreOrder, destinationSubmitOrder, destinationCloseOrder,
                destinationRestoreOrder);
    }

    @Override
    public String toString() {
        return "OperationOrdersModel{" +
                "submitOrder=" + submitOrder +
                ", closeOrder=" + closeOrder +
                ", restoreOrder=" + restoreOrder +
                ", destinationSubmitOrder=" + destinationSubmitOrder +
                ", destinationCloseOrder=" + destinationCloseOrder +
                ", destinationRestoreOrder=" + destinationRestoreOrder +
                '}';
    }

    public static final class Builder {

        private Long submitOrder;
        private Long closeOrder;
        private Long restoreOrder;
        private Long destinationSubmitOrder;
        private Long destinationCloseOrder;
        private Long destinationRestoreOrder;

        private Builder() {
        }

        private Builder(OperationOrdersModel value) {
            this.submitOrder = value.submitOrder;
            this.closeOrder = value.closeOrder;
            this.restoreOrder = value.restoreOrder;
            this.destinationSubmitOrder = value.destinationSubmitOrder;
            this.destinationCloseOrder = value.destinationCloseOrder;
            this.destinationRestoreOrder = value.destinationRestoreOrder;
        }

        public Builder submitOrder(long submitOrder) {
            this.submitOrder = submitOrder;
            return this;
        }

        public Builder closeOrder(Long closeOrder) {
            this.closeOrder = closeOrder;
            return this;
        }

        public Builder restoreOrder(Long restoreOrder) {
            this.restoreOrder = restoreOrder;
            return this;
        }

        public Builder destinationSubmitOrder(Long destinationSubmitOrder) {
            this.destinationSubmitOrder = destinationSubmitOrder;
            return this;
        }

        public Builder destinationCloseOrder(Long destinationCloseOrder) {
            this.destinationCloseOrder = destinationCloseOrder;
            return this;
        }

        public Builder destinationRestoreOrder(Long destinationRestoreOrder) {
            this.destinationRestoreOrder = destinationRestoreOrder;
            return this;
        }

        public OperationOrdersModel build() {
            Preconditions.checkNotNull(submitOrder, "SubmitOrder is required");
            return new OperationOrdersModel(submitOrder, closeOrder, restoreOrder, destinationSubmitOrder,
                    destinationCloseOrder, destinationRestoreOrder);
        }

    }

}
