package ru.yandex.intranet.d.model.delivery;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;

/**
 * Deliverable metadata history.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliverableMetaHistoryModel {

    private final long quotaRequestId;
    private final long campaignId;
    private final Set<Long> bigOrderIds;
    private final String deliveryId;

    @JsonCreator
    public DeliverableMetaHistoryModel(long quotaRequestId, long campaignId, Set<Long> bigOrderIds, Long bigOrderId,
                                       String deliveryId) {
        this.quotaRequestId = quotaRequestId;
        this.campaignId = campaignId;
        this.deliveryId = deliveryId;
        this.bigOrderIds = new HashSet<>();
        if (bigOrderIds != null) {
            this.bigOrderIds.addAll(bigOrderIds);
        }
        if (bigOrderId != null) {
            this.bigOrderIds.add(bigOrderId);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public long getQuotaRequestId() {
        return quotaRequestId;
    }

    public long getCampaignId() {
        return campaignId;
    }

    public Set<Long> getBigOrderIds() {
        return bigOrderIds;
    }

    public String getDeliveryId() {
        return deliveryId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableMetaHistoryModel that = (DeliverableMetaHistoryModel) o;
        return quotaRequestId == that.quotaRequestId && campaignId == that.campaignId && Objects.equals(bigOrderIds,
                that.bigOrderIds) && Objects.equals(deliveryId, that.deliveryId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(quotaRequestId, campaignId, bigOrderIds, deliveryId);
    }

    @Override
    public String toString() {
        return "DeliverableMetaHistoryModel{" +
                "quotaRequestId=" + quotaRequestId +
                ", campaignId=" + campaignId +
                ", bigOrderIds=" + bigOrderIds +
                ", deliveryId='" + deliveryId + '\'' +
                '}';
    }

    public static final class Builder {

        private Long quotaRequestId;
        private Long campaignId;
        private final Set<Long> bigOrderIds = new HashSet<>();
        private String deliveryId;

        private Builder() {
        }

        public Builder quotaRequestId(long quotaRequestId) {
            this.quotaRequestId = quotaRequestId;
            return this;
        }

        public Builder campaignId(long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        public Builder bigOrderIds(long bigOrderId) {
            this.bigOrderIds.add(bigOrderId);
            return this;
        }

        public Builder bigOrderIds(Collection<Long> bigOrderIds) {
            this.bigOrderIds.addAll(bigOrderIds);
            return this;
        }

        public Builder deliveryId(String deliveryId) {
            this.deliveryId = deliveryId;
            return this;
        }

        public DeliverableMetaHistoryModel build() {
            if (bigOrderIds.isEmpty()) {
                throw new IllegalArgumentException("BigOrderId is required");
            }

            Preconditions.checkNotNull(quotaRequestId, "QuotaRequestId is required");
            Preconditions.checkNotNull(campaignId, "CampaignId is required");
            Preconditions.checkNotNull(deliveryId, "DeliveryId is required");
            return new DeliverableMetaHistoryModel(quotaRequestId, campaignId, bigOrderIds, null, deliveryId);
        }

    }

}
