package ru.yandex.intranet.d.model.delivery;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.base.Preconditions;

/**
 * Deliverable request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliverableRequestModel {

    private final Long serviceId;
    private final String folderId;
    private final String resourceId;
    private final DeliverableDeltaModel delta;
    private final DeliverableMetaRequestModel meta;

    @JsonCreator
    public DeliverableRequestModel(Long serviceId, String folderId, String resourceId, DeliverableDeltaModel delta,
                                   DeliverableMetaRequestModel meta) {
        this.serviceId = serviceId;
        this.folderId = folderId;
        this.resourceId = resourceId;
        this.delta = delta;
        this.meta = meta;
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<Long> getServiceId() {
        return Optional.ofNullable(serviceId);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    public String getResourceId() {
        return resourceId;
    }

    public DeliverableDeltaModel getDelta() {
        return delta;
    }

    public DeliverableMetaRequestModel getMeta() {
        return meta;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableRequestModel that = (DeliverableRequestModel) o;
        return Objects.equals(serviceId, that.serviceId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(delta, that.delta) &&
                Objects.equals(meta, that.meta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(serviceId, folderId, resourceId, delta, meta);
    }

    @Override
    public String toString() {
        return "DeliverableRequestModel{" +
                "serviceId=" + serviceId +
                ", folderId='" + folderId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", delta=" + delta +
                ", meta=" + meta +
                '}';
    }

    public static final class Builder {

        private Long serviceId;
        private String folderId;
        private String resourceId;
        private DeliverableDeltaModel delta;
        private DeliverableMetaRequestModel meta;

        private Builder() {
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder delta(DeliverableDeltaModel delta) {
            this.delta = delta;
            return this;
        }

        public Builder meta(DeliverableMetaRequestModel meta) {
            this.meta = meta;
            return this;
        }

        public DeliverableRequestModel build() {
            Preconditions.checkNotNull(resourceId, "ResourceId is required");
            Preconditions.checkNotNull(delta, "Delta is required");
            Preconditions.checkNotNull(meta, "Meta is required");
            Preconditions.checkArgument(serviceId != null || folderId != null,
                    "Either ServiceId or FolderId is required");
            return new DeliverableRequestModel(serviceId, folderId, resourceId, delta, meta);
        }

    }

}
