package ru.yandex.intranet.d.model.delivery;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.base.Preconditions;

/**
 * Deliverable response.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliverableResponseModel {

    private final Long serviceId;
    private final String folderId;
    private final Set<String> resourceIds;
    private final DeliverableFolderOperationModel folderOperationLog;
    private final DeliverableMetaResponseModel meta;

    @JsonCreator
    public DeliverableResponseModel(Long serviceId, String folderId, Set<String> resourceIds,
                                    DeliverableFolderOperationModel folderOperationLog,
                                    DeliverableMetaResponseModel meta) {
        this.serviceId = serviceId;
        this.folderId = folderId;
        this.resourceIds = resourceIds;
        this.folderOperationLog = folderOperationLog;
        this.meta = meta;
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<Long> getServiceId() {
        return Optional.ofNullable(serviceId);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    public Set<String> getResourceIds() {
        return resourceIds;
    }

    public DeliverableFolderOperationModel getFolderOperationLog() {
        return folderOperationLog;
    }

    public DeliverableMetaResponseModel getMeta() {
        return meta;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableResponseModel that = (DeliverableResponseModel) o;
        return Objects.equals(serviceId, that.serviceId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceIds, that.resourceIds) &&
                Objects.equals(folderOperationLog, that.folderOperationLog) &&
                Objects.equals(meta, that.meta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(serviceId, folderId, resourceIds, folderOperationLog, meta);
    }

    @Override
    public String toString() {
        return "DeliverableResponseModel{" +
                "serviceId=" + serviceId +
                ", folderId='" + folderId + '\'' +
                ", resourceIds='" + resourceIds + '\'' +
                ", folderOperationLog=" + folderOperationLog +
                ", meta=" + meta +
                '}';
    }

    public static final class Builder {

        private final Set<String> resourceIds = new HashSet<>();

        private Long serviceId;
        private String folderId;
        private DeliverableFolderOperationModel folderOperationLog;
        private DeliverableMetaResponseModel meta;

        private Builder() {
        }

        public Builder serviceId(Long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder addResourceId(String resourceId) {
            this.resourceIds.add(resourceId);
            return this;
        }

        public Builder addResourceIds(Collection<? extends String> resourceIds) {
            this.resourceIds.addAll(resourceIds);
            return this;
        }

        public Builder folderOperationLog(DeliverableFolderOperationModel folderOperationLog) {
            this.folderOperationLog = folderOperationLog;
            return this;
        }

        public Builder meta(DeliverableMetaResponseModel meta) {
            this.meta = meta;
            return this;
        }

        public DeliverableResponseModel build() {
            Preconditions.checkNotNull(folderOperationLog, "FolderOperationLog is required");
            Preconditions.checkNotNull(meta, "Meta is required");
            Preconditions.checkArgument(serviceId != null || folderId != null,
                    "Either ServiceId or FolderId is required");
            return new DeliverableResponseModel(serviceId, folderId, resourceIds, folderOperationLog, meta);
        }
    }

}
