package ru.yandex.intranet.d.model.delivery;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;

/**
 * Delivery request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliveryRequestModel {

    private final String deliveryId;
    private final String authorUid;
    private final Set<DeliverableRequestModel> deliverables;

    @JsonCreator
    public DeliveryRequestModel(String deliveryId, String authorUid, Set<DeliverableRequestModel> deliverables) {
        this.deliveryId = deliveryId;
        this.authorUid = authorUid;
        this.deliverables = deliverables;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getDeliveryId() {
        return deliveryId;
    }

    public String getAuthorUid() {
        return authorUid;
    }

    public Set<DeliverableRequestModel> getDeliverables() {
        return deliverables;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliveryRequestModel that = (DeliveryRequestModel) o;
        return Objects.equals(deliveryId, that.deliveryId) &&
                Objects.equals(authorUid, that.authorUid) &&
                Objects.equals(deliverables, that.deliverables);
    }

    @Override
    public int hashCode() {
        return Objects.hash(deliveryId, authorUid, deliverables);
    }

    @Override
    public String toString() {
        return "DeliveryRequestModel{" +
                "deliveryId='" + deliveryId + '\'' +
                ", authorUid='" + authorUid + '\'' +
                ", deliverables=" + deliverables +
                '}';
    }

    public static final class Builder {

        private String deliveryId;
        private String authorUid;
        private final Set<DeliverableRequestModel> deliverables = new HashSet<>();

        private Builder() {
        }

        public Builder deliveryId(String deliveryId) {
            this.deliveryId = deliveryId;
            return this;
        }

        public Builder authorUid(String authorUid) {
            this.authorUid = authorUid;
            return this;
        }

        public Builder addDeliverable(DeliverableRequestModel deliverable) {
            this.deliverables.add(deliverable);
            return this;
        }

        public DeliveryRequestModel build() {
            Preconditions.checkNotNull(deliveryId, "DeliveryId is required");
            Preconditions.checkNotNull(authorUid, "AuthorUid is required");
            return new DeliveryRequestModel(deliveryId, authorUid, deliverables);
        }

    }

}
