package ru.yandex.intranet.d.model.folders;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import ru.yandex.intranet.d.model.accounts.AccountReserveType;

/**
 * Account history model.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public final class AccountHistoryModel {

    private final Long version;
    private final String providerId;
    private final String outerAccountIdInProvider;
    private final String outerAccountKeyInProvider;
    private final String folderId;
    private final String displayName;
    private final Boolean deleted;
    private final Long lastReceivedVersion;
    private final String accountsSpacesId;
    private final Boolean freeTier;
    @Nullable
    private final AccountReserveType reserveType;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public AccountHistoryModel(@JsonProperty("version") Long version,
                               @JsonProperty("providerId") String providerId,
                               @JsonProperty("outerAccountIdInProvider") String outerAccountIdInProvider,
                               @JsonProperty("outerAccountKeyInProvider") String outerAccountKeyInProvider,
                               @JsonProperty("folderId") String folderId,
                               @JsonProperty("displayName") String displayName,
                               @JsonProperty("deleted") Boolean deleted,
                               @JsonProperty("lastReceivedVersion") Long lastReceivedVersion,
                               @JsonProperty("accountsSpacesId") String accountsSpacesId,
                               @JsonProperty("freeTier") Boolean freeTier,
                               @JsonProperty("reserveType") @Nullable AccountReserveType reserveType
    ) {
        this.version = version;
        this.providerId = providerId;
        this.outerAccountIdInProvider = outerAccountIdInProvider;
        this.outerAccountKeyInProvider = outerAccountKeyInProvider;
        this.folderId = folderId;
        this.displayName = displayName;
        this.deleted = deleted;
        this.lastReceivedVersion = lastReceivedVersion;
        this.accountsSpacesId = accountsSpacesId;
        this.freeTier = freeTier;
        this.reserveType = reserveType;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Optional<Long> getVersion() {
        return Optional.ofNullable(version);
    }

    public Optional<String> getProviderId() {
        return Optional.ofNullable(providerId);
    }

    public Optional<String> getOuterAccountIdInProvider() {
        return Optional.ofNullable(outerAccountIdInProvider);
    }

    public Optional<String> getOuterAccountKeyInProvider() {
        return Optional.ofNullable(outerAccountKeyInProvider);
    }

    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    public Optional<String> getDisplayName() {
        return Optional.ofNullable(displayName);
    }

    public Optional<Boolean> getDeleted() {
        return Optional.ofNullable(deleted);
    }

    public Optional<Long> getLastReceivedVersion() {
        return Optional.ofNullable(lastReceivedVersion);
    }

    public String getAccountsSpacesId() {
        return accountsSpacesId;
    }

    @JsonProperty("freeTier")
    public Optional<Boolean> isFreeTier() {
        return Optional.ofNullable(freeTier);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<AccountReserveType> getReserveType() {
        return Optional.ofNullable(reserveType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountHistoryModel that = (AccountHistoryModel) o;
        return Objects.equals(freeTier, that.freeTier) &&
                Objects.equals(version, that.version) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(outerAccountIdInProvider, that.outerAccountIdInProvider) &&
                Objects.equals(outerAccountKeyInProvider, that.outerAccountKeyInProvider) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(deleted, that.deleted) &&
                Objects.equals(lastReceivedVersion, that.lastReceivedVersion) &&
                Objects.equals(accountsSpacesId, that.accountsSpacesId) &&
                Objects.equals(reserveType, that.reserveType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(version, providerId, outerAccountIdInProvider, outerAccountKeyInProvider, folderId,
                displayName, deleted, lastReceivedVersion, accountsSpacesId, freeTier, reserveType);
    }

    @Override
    public String toString() {
        return "AccountHistoryModel{" +
                "version=" + version +
                ", providerId='" + providerId + '\'' +
                ", outerAccountIdInProvider='" + outerAccountIdInProvider + '\'' +
                ", outerAccountKeyInProvider='" + outerAccountKeyInProvider + '\'' +
                ", folderId='" + folderId + '\'' +
                ", displayName='" + displayName + '\'' +
                ", deleted=" + deleted +
                ", lastReceivedVersion=" + lastReceivedVersion +
                ", accountsSpacesId='" + accountsSpacesId + '\'' +
                ", freeTier=" + freeTier +
                ", reserveType=" + reserveType +
                '}';
    }

    public static class Builder {

        private Long version;
        private String providerId;
        private String outerAccountIdInProvider;
        private String outerAccountKeyInProvider;
        private String folderId;
        private String displayName;
        private Boolean deleted;
        private Long lastReceivedVersion;
        private String accountsSpacesId;
        private Boolean freeTier;
        private AccountReserveType reserveType;

        private Builder() {
        }

        public Builder version(Long version) {
            this.version = version;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder outerAccountIdInProvider(String outerAccountIdInProvider) {
            this.outerAccountIdInProvider = outerAccountIdInProvider;
            return this;
        }

        public Builder outerAccountKeyInProvider(String outerAccountKeyInProvider) {
            this.outerAccountKeyInProvider = outerAccountKeyInProvider;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder deleted(Boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder lastReceivedVersion(Long lastReceivedVersion) {
            this.lastReceivedVersion = lastReceivedVersion;
            return this;
        }

        public Builder accountsSpacesId(String accountsSpacesId) {
            this.accountsSpacesId = accountsSpacesId;
            return this;
        }

        public Builder freeTier(Boolean freeTier) {
            this.freeTier = freeTier;
            return this;
        }

        public Builder reserveType(@Nullable AccountReserveType reserveType) {
            this.reserveType = reserveType;
            return this;
        }

        public AccountHistoryModel build() {
            return new AccountHistoryModel(version, providerId, outerAccountIdInProvider, outerAccountKeyInProvider,
                    folderId, displayName, deleted, lastReceivedVersion, accountsSpacesId, freeTier, reserveType);
        }

    }

}
